package org.nuclearfog.smither.backend.async;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.nuclearfog.smither.backend.api.Connection;
import org.nuclearfog.smither.backend.api.ConnectionException;
import org.nuclearfog.smither.backend.api.ConnectionManager;
import org.nuclearfog.smither.model.lists.UserLists;

/**
 * Background task for downloading  user lists created by a user
 *
 * @author nuclearfog
 * @see org.nuclearfog.smither.ui.fragments.UserListFragment
 */
public class UserListLoader extends AsyncExecutor<UserListLoader.Param, UserListLoader.Result> {

	private Connection connection;

	/**
	 *
	 */
	public UserListLoader(Context context) {
		connection = ConnectionManager.getDefaultConnection(context);
	}


	@Override
	protected Result doInBackground(@NonNull Param param) {
		try {
			switch (param.type) {
				case Param.OWNERSHIP:
					UserLists userlists = connection.getUserListOwnerships(param.id, param.cursor);
					return new Result(param.index, userlists, null);

				case Param.MEMBERSHIP:
					userlists = connection.getUserListMemberships(param.id, param.cursor);
					return new Result(param.index, userlists, null);

				default:
					return null;
			}
		} catch (ConnectionException exception) {
			return new Result(param.index, null, exception);
		}
	}

	/**
	 *
	 */
	public static class Param {

		public static final long NO_CURSOR = -1L;

		public static final int OWNERSHIP = 1;
		public static final int MEMBERSHIP = 2;

		final int type, index;
		final long id, cursor;

		/**
		 * @param type   type of user lists to load
		 * @param index  index where to insert new items in the list/adapter
		 * @param id     user list ID
		 * @param cursor cursor to parse the results
		 */
		public Param(int type, int index, long id, long cursor) {
			this.type = type;
			this.id = id;
			this.index = index;
			this.cursor = cursor;
		}
	}

	/**
	 *
	 */
	public static class Result {

		public final int index;
		@Nullable
		public final UserLists userlists;
		@Nullable
		public final ConnectionException exception;

		/**
		 * @param index     index where to insert new items in the list/adapter
		 * @param userLists result or null if an error occurred
		 * @param exception if not null an error occurred
		 */
		Result(int index, @Nullable UserLists userLists, @Nullable ConnectionException exception) {
			this.userlists = userLists;
			this.exception = exception;
			this.index = index;
		}
	}
}