package org.nuclearfog.smither.backend.async;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.text.Spannable;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.nuclearfog.smither.backend.api.Connection;
import org.nuclearfog.smither.backend.api.ConnectionManager;
import org.nuclearfog.smither.backend.helper.MediaStatus;
import org.nuclearfog.smither.backend.image.EmojiCache;
import org.nuclearfog.smither.model.Emoji;

import java.io.InputStream;
import java.util.Map;
import java.util.TreeMap;

/**
 * Emoji image loader class
 *
 * @author nuclearfog
 */
public class TextEmojiLoader extends AsyncExecutor<TextEmojiLoader.Param, TextEmojiLoader.Result> {

	private Connection connection;
	private EmojiCache cache;

	/**
	 *
	 */
	public TextEmojiLoader(Context context) {
		connection = ConnectionManager.getDefaultConnection(context);
		cache = EmojiCache.get(context);
	}


	@Override
	protected Result doInBackground(@NonNull Param param) {
		try {
			Map<String, Bitmap> result = new TreeMap<>();
			for (Emoji emoji : param.emojis) {
				Bitmap icon = cache.getImage(emoji.getUrl());
				if (icon == null) {
					MediaStatus media = connection.downloadImage(emoji.getUrl());
					InputStream input = media.getStream();
					icon = BitmapFactory.decodeStream(input);
					cache.putImage(emoji.getUrl(), icon);
				}
				if (icon.getHeight() > 0 && icon.getWidth() > 0) {
					icon = Bitmap.createScaledBitmap(icon, icon.getWidth() * param.size / icon.getHeight(), param.size, false);
				} else {
					icon = Bitmap.createScaledBitmap(icon, param.size, param.size, false);
				}
				result.put(emoji.getCode(), icon);
			}
			cache.trimCache();
			return new Result(param.id, param.spannable, result);
		} catch (Exception exception) {
			return new Result(param.id, param.spannable, null);
		}
	}

	/**
	 *
	 */
	public static class Param {

		final Emoji[] emojis;
		final Spannable spannable;
		final long id;
		final int size;

		/**
		 * @param emojis    array of emojis
		 * @param spannable spannable string containing emoji tags
		 * @param size      icon size of the emoji picture
		 */
		public Param(Emoji[] emojis, Spannable spannable, int size) {
			this(0L, emojis, spannable, size);
		}

		/**
		 * @param id        ID used to identify task (used if multiple tasks are running at the same time)
		 * @param emojis    array of emojis
		 * @param spannable spannable string containing emoji tags
		 * @param size      icon size of the emoji picture
		 */
		public Param(long id, Emoji[] emojis, Spannable spannable, int size) {
			this.emojis = emojis;
			this.spannable = spannable;
			this.size = size;
			this.id = id;
		}
	}

	/**
	 *
	 */
	public static class Result {

		@Nullable
		public final Map<String, Bitmap> images;
		public final Spannable spannable;
		public final long id;

		/**
		 * @param id        ID of the task
		 * @param spannable spannable string containing emoji tags
		 * @param images    a map containing emoji tags and images
		 */
		Result(long id, Spannable spannable, @Nullable Map<String, Bitmap> images) {
			this.images = images;
			this.spannable = spannable;
			this.id = id;
		}
	}
}