package org.nuclearfog.smither.backend.api.mastodon.impl;

import android.util.Patterns;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.jsoup.Jsoup;
import org.nuclearfog.smither.model.Emoji;
import org.nuclearfog.smither.model.Field;
import org.nuclearfog.smither.model.User;
import org.nuclearfog.smither.utils.StringUtils;

/**
 * User implementation for Mastodon API
 *
 * @author nuclearfog
 */
public class MastodonUser implements User {

	private static final long serialVersionUID = 7322491410965084755L;

	private long id;
	private long createdAt;
	private String url;
	private String screenName;
	private String username;
	private String profileUrl = "";
	private String bannerUrl = "";
	private String description = "";

	private int following;
	private int follower;
	private int statusCount;
	private boolean locked;
	private boolean isCurrentUser;
	private boolean indexable;
	private boolean discoverable;
	private boolean group;
	private boolean bot;
	private Emoji[] emojis = {};
	private Field[] fields = {};

	/**
	 * default constructor for all user instances
	 *
	 * @param json          json object used by Mastodon API
	 * @param currentUserId current user ID
	 */
	public MastodonUser(JSONObject json, long currentUserId) throws JSONException {
		JSONArray emojiArray = json.optJSONArray("emojis");
		JSONArray fieldsArray = json.optJSONArray("fields");
		String idStr = json.getString("id");
		String description = json.optString("note", "");
		String profileUrl = json.optString("avatar", "");
		String bannerUrl = json.optString("header", "");
		screenName = '@' + json.optString("acct", "");
		username = json.optString("display_name", "");
		createdAt = StringUtils.getIsoTime(json.optString("created_at", ""));
		url = json.optString("url", "");
		following = json.optInt("following_count");
		follower = json.optInt("followers_count");
		statusCount = json.optInt("statuses_count");
		locked = json.optBoolean("locked");
		indexable = !json.optBoolean("indexable", true);
		discoverable = json.optBoolean("discoverable", false);
		bot = json.optBoolean("bot", false);
		group = json.optBoolean("group", false);
		if (!description.isEmpty()) {
			this.description = Jsoup.parse(description).text();
		}
		if (Patterns.WEB_URL.matcher(profileUrl).matches()) {
			this.profileUrl = profileUrl;
		}
		if (Patterns.WEB_URL.matcher(bannerUrl).matches()) {
			this.bannerUrl = bannerUrl;
		}
		if (emojiArray != null && emojiArray.length() > 0) {
			emojis = new Emoji[emojiArray.length()];
			for (int i = 0; i < emojis.length; i++) {
				JSONObject emojiJson = emojiArray.getJSONObject(i);
				emojis[i] = new MastodonEmoji(emojiJson);
			}
		}
		if (fieldsArray != null && fieldsArray.length() > 0) {
			fields = new Field[fieldsArray.length()];
			for (int i = 0; i < fields.length; i++) {
				JSONObject fieldJson = fieldsArray.getJSONObject(i);
				fields[i] = new MastodonField(fieldJson);
			}
		}
		try {
			id = Long.parseLong(idStr);
			isCurrentUser = currentUserId == id;
		} catch (NumberFormatException e) {
			throw new JSONException("bad user ID:" + idStr);
		}
	}


	@Override
	public long getId() {
		return id;
	}


	@Override
	public String getUsername() {
		return username;
	}


	@Override
	public String getScreenName() {
		return screenName;
	}


	@Override
	public long getTimestamp() {
		return createdAt;
	}


	@Override
	public String getOriginalProfileImageUrl() {
		return profileUrl;
	}


	@Override
	public String getProfileImageThumbnailUrl() {
		return profileUrl; // todo switch to thumbnail url if supported by API
	}


	@Override
	public String getOriginalBannerImageUrl() {
		return bannerUrl;
	}


	@Override
	public String getBannerImageThumbnailUrl() {
		return bannerUrl; // todo switch to thumbnail url if supported by API
	}


	@Override
	public String getDescription() {
		return description;
	}


	@Override
	public String getLocation() {
		return "";
	}


	@Override
	public String getProfileUrl() {
		return url;
	}


	@Override
	public boolean isVerified() {
		// not supported
		return false;
	}


	@Override
	public boolean isProtected() {
		return locked;
	}


	@Override
	public boolean isIndexable() {
		return indexable;
	}


	@Override
	public boolean isDiscoverable() {
		return discoverable;
	}


	@Override
	public boolean isBot() {
		return bot;
	}


	@Override
	public boolean isGroup() {
		return group;
	}


	@Override
	public int getFollowing() {
		return following;
	}


	@Override
	public int getFollower() {
		return follower;
	}


	@Override
	public int getStatusCount() {
		return statusCount;
	}


	@Override
	public int getFavoriteCount() {
		return -1;
	}


	@Override
	public boolean hasDefaultProfileImage() {
		return false;
	}


	@Override
	public boolean isCurrentUser() {
		return isCurrentUser;
	}


	@Override
	public Emoji[] getEmojis() {
		return emojis;
	}


	@Override
	public Field[] getFields() {
		return fields;
	}


	@Override
	public boolean equals(@Nullable Object obj) {
		if (!(obj instanceof User))
			return false;
		return ((User) obj).getId() == getId();
	}


	@NonNull
	@Override
	public String toString() {
		return "name=\"" + getScreenName() + "\"";
	}
}