package org.nuclearfog.apollo.utils;

import android.content.Context;
import android.util.Log;

import org.nuclearfog.apollo.R;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * This class contains utils for strings
 *
 * @author nuclearfog
 */
public final class StringUtils {

	private static final String TAG = "StringUtils";

	/* This class is never initiated */
	private StringUtils() {
	}

	/**
	 * Used to create a formatted time string for the duration of tracks.
	 *
	 * @param duration The track in milliseconds.
	 * @return Duration of a track that's properly formatted.
	 */
	public static String makeTimeString(Context context, long duration) {
		if (duration < 0) {
			// invalid time
			return "--:--";
		}
		long sec = duration / 1000;
		long min = sec / 60;
		long hour = min / 60;
		if (hour > 0)
			return String.format(context.getString(R.string.duration_format_long), hour, min % 60, sec % 60);
		return String.format(context.getString(R.string.duration_format_short), min % 60, sec % 60);
	}

	/**
	 * Used to make number of labels for the number of artists, albums, songs,
	 * genres, and playlists.
	 *
	 * @param pluralInt The ID of the plural string to use.
	 * @param number    The number of artists, albums, songs, genres, or playlists.
	 * @return A {@link String} used as a label for the number of artists,
	 * albums, songs, genres, and playlists.
	 */
	public static String makeLabel(Context context, int pluralInt, int number) {
		return context.getResources().getQuantityString(pluralInt, number, number);
	}

	/**
	 * encode text o UTF-8
	 *
	 * @param text text to encode
	 * @return UTF8 translated string
	 */
	public static String encodeUTF8(String text) {
		try {
			return URLEncoder.encode(text, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			return "";
		}
	}

	/**
	 * A hashing method that changes a string (like a URL) into a hash suitable
	 * for using as a disk filename.
	 *
	 * @param key The key used to store the file
	 */
	public static String hashKeyForDisk(String key) {
		String cacheKey;
		try {
			MessageDigest digest = MessageDigest.getInstance("MD5");
			digest.update(key.getBytes());
			StringBuilder builder = new StringBuilder();
			for (byte b : digest.digest()) {
				String hex = Integer.toHexString(0xFF & b);
				if (hex.length() == 1) {
					builder.append('0');
				}
				builder.append(hex);
			}
			cacheKey = builder.toString();
		} catch (NoSuchAlgorithmException e) {
			Log.e(TAG, "hashKeyForDisk()", e);
			// use fallback hash code
			cacheKey = String.valueOf(key.hashCode());
		}
		return cacheKey;
	}

	/**
	 * get file/folder name of a path string
	 *
	 * @param str path to a file
	 * @return file/folder name or str parameter if it's not a path
	 */
	public static String getFolderName(String str) {
		if (str.length() > 2 && str.startsWith("/")) {
			if (str.endsWith("/"))
				str = str.substring(0, str.length() - 1);
			int div = str.lastIndexOf('/');
			if (div > 0) {
				return str.substring(div + 1);
			}
		}
		return str;
	}

	/**
	 * create unique cache key for specific entry
	 *
	 * @param type image type to cache
	 * @param ids  at least one ID to identify
	 * @return key string
	 */
	public static String generateCacheKey(Constants.ImageType type, long... ids) {
		StringBuilder str = new StringBuilder();
		for (long id : ids) {
			str.append(id).append('_');
		}
		str.append(type.value);
		return str.toString();
	}

	/**
	 * create unique cache key for specific entry
	 *
	 * @param type image type to cache
	 * @param mbid MusicBrainz ID
	 * @return key string
	 */
	public static String generateCacheKey(Constants.ImageType type, String mbid) {
		return type.value + "_" + mbid;
	}
}