package org.nuclearfog.apollo.utils;

import android.app.ActivityManager;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.TransitionDrawable;

/**
 * {@link Bitmap} specific helpers.
 *
 * @author Andrew Neal (andrewdneal@gmail.com)
 */
public final class ImageUtils {

	/**
	 * Initial blur radius.
	 */
	private static final int DEFAULT_BLUR_RADIUS = 8;

	/**
	 * Default transition drawable fade time
	 */
	private static final int FADE_IN_TIME = 200;

	/**
	 * This class is never instantiated
	 */
	private ImageUtils() {
	}

	/**
	 * Takes a bitmap and creates a new slightly blurry version of it.
	 *
	 * @param sentBitmap The {@link Bitmap} to blur.
	 * @return A blurred version of the given {@link Bitmap}.
	 */
	public static Bitmap createBlurredBitmap(Bitmap sentBitmap) {
		if (sentBitmap == null || sentBitmap.getConfig() == null) {
			return null;
		}

		// Stack Blur v1.0 from
		// http://www.quasimondo.com/StackBlurForCanvas/StackBlurDemo.html
		//
		// Java Author: Mario Klingemann <mario at quasimondo.com>
		// http://incubator.quasimondo.com
		// created February 29, 2004
		// Android port : Yahel Bouaziz <yahel at kayenko.com>
		// http://www.kayenko.com
		// ported april 5th, 2012

		// This is a compromise between Gaussian Blur and Box blur
		// It creates much better looking blurs than Box Blur, but is
		// 7x faster than my Gaussian Blur implementation.
		//
		// I called it Stack Blur because this describes best how this
		// filter works internally: it creates a kind of moving stack
		// of colors whilst scanning through the image. Thereby it
		// just has to add one new block of color to the right side
		// of the stack and remove the leftmost color. The remaining
		// colors on the topmost layer of the stack are either added on
		// or reduced by one, depending on if they are on the right or
		// on the left side of the stack.
		//
		// If you are using this algorithm in your code please add
		// the following line:
		//
		// Stack Blur Algorithm by Mario Klingemann <mario@quasimondo.com>

		Bitmap mBitmap = sentBitmap.copy(sentBitmap.getConfig(), true);

		int w = mBitmap.getWidth();
		int h = mBitmap.getHeight();

		int[] pix = new int[w * h];
		mBitmap.getPixels(pix, 0, w, 0, 0, w, h);

		int wm = w - 1;
		int hm = h - 1;
		int wh = w * h;
		int div = DEFAULT_BLUR_RADIUS + DEFAULT_BLUR_RADIUS + 1;

		int[] r = new int[wh];
		int[] g = new int[wh];
		int[] b = new int[wh];
		int[] vmin = new int[Math.max(w, h)];
		int rsum, gsum, bsum, x, y, i, p, yp, yi, yw;

		int divsum = div + 1 >> 1;
		divsum *= divsum;
		int[] dv = new int[256 * divsum];
		for (i = 0; i < 256 * divsum; i++) {
			dv[i] = i / divsum;
		}

		yw = yi = 0;

		int[][] stack = new int[div][3];
		int stackpointer;
		int stackstart;
		int[] sir;
		int rbs;
		int r1 = DEFAULT_BLUR_RADIUS + 1;
		int routsum, goutsum, boutsum;
		int rinsum, ginsum, binsum;

		for (y = 0; y < h; y++) {
			rinsum = ginsum = binsum = routsum = goutsum = boutsum = rsum = gsum = bsum = 0;
			for (i = -DEFAULT_BLUR_RADIUS; i <= DEFAULT_BLUR_RADIUS; i++) {
				p = pix[yi + Math.min(wm, Math.max(i, 0))];
				sir = stack[i + DEFAULT_BLUR_RADIUS];
				sir[0] = (p & 0xff0000) >> 16;
				sir[1] = (p & 0x00ff00) >> 8;
				sir[2] = p & 0x0000ff;
				rbs = r1 - Math.abs(i);
				rsum += sir[0] * rbs;
				gsum += sir[1] * rbs;
				bsum += sir[2] * rbs;
				if (i > 0) {
					rinsum += sir[0];
					ginsum += sir[1];
					binsum += sir[2];
				} else {
					routsum += sir[0];
					goutsum += sir[1];
					boutsum += sir[2];
				}
			}
			stackpointer = DEFAULT_BLUR_RADIUS;

			for (x = 0; x < w; x++) {

				r[yi] = dv[rsum];
				g[yi] = dv[gsum];
				b[yi] = dv[bsum];

				rsum -= routsum;
				gsum -= goutsum;
				bsum -= boutsum;

				stackstart = stackpointer - DEFAULT_BLUR_RADIUS + div;
				sir = stack[stackstart % div];

				routsum -= sir[0];
				goutsum -= sir[1];
				boutsum -= sir[2];

				if (y == 0) {
					vmin[x] = Math.min(x + DEFAULT_BLUR_RADIUS + 1, wm);
				}
				p = pix[yw + vmin[x]];

				sir[0] = (p & 0xff0000) >> 16;
				sir[1] = (p & 0x00ff00) >> 8;
				sir[2] = p & 0x0000ff;

				rinsum += sir[0];
				ginsum += sir[1];
				binsum += sir[2];

				rsum += rinsum;
				gsum += ginsum;
				bsum += binsum;

				stackpointer = (stackpointer + 1) % div;
				sir = stack[stackpointer % div];

				routsum += sir[0];
				goutsum += sir[1];
				boutsum += sir[2];

				rinsum -= sir[0];
				ginsum -= sir[1];
				binsum -= sir[2];

				yi++;
			}
			yw += w;
		}
		for (x = 0; x < w; x++) {
			rinsum = ginsum = binsum = routsum = goutsum = boutsum = rsum = gsum = bsum = 0;
			yp = -DEFAULT_BLUR_RADIUS * w;
			for (i = -DEFAULT_BLUR_RADIUS; i <= DEFAULT_BLUR_RADIUS; i++) {
				yi = Math.max(0, yp) + x;

				sir = stack[i + DEFAULT_BLUR_RADIUS];

				sir[0] = r[yi];
				sir[1] = g[yi];
				sir[2] = b[yi];

				rbs = r1 - Math.abs(i);

				rsum += r[yi] * rbs;
				gsum += g[yi] * rbs;
				bsum += b[yi] * rbs;

				if (i > 0) {
					rinsum += sir[0];
					ginsum += sir[1];
					binsum += sir[2];
				} else {
					routsum += sir[0];
					goutsum += sir[1];
					boutsum += sir[2];
				}

				if (i < hm) {
					yp += w;
				}
			}
			yi = x;
			stackpointer = DEFAULT_BLUR_RADIUS;
			for (y = 0; y < h; y++) {
				pix[yi] = 0xff000000 | dv[rsum] << 16 | dv[gsum] << 8 | dv[bsum];

				rsum -= routsum;
				gsum -= goutsum;
				bsum -= boutsum;

				stackstart = stackpointer - DEFAULT_BLUR_RADIUS + div;
				sir = stack[stackstart % div];

				routsum -= sir[0];
				goutsum -= sir[1];
				boutsum -= sir[2];

				if (x == 0) {
					vmin[y] = Math.min(y + r1, hm) * w;
				}
				p = x + vmin[y];

				sir[0] = r[p];
				sir[1] = g[p];
				sir[2] = b[p];

				rinsum += sir[0];
				ginsum += sir[1];
				binsum += sir[2];

				rsum += rinsum;
				gsum += ginsum;
				bsum += binsum;

				stackpointer = (stackpointer + 1) % div;
				sir = stack[stackpointer];

				routsum += sir[0];
				goutsum += sir[1];
				boutsum += sir[2];

				rinsum -= sir[0];
				ginsum -= sir[1];
				binsum -= sir[2];

				yi += w;
			}
		}

		mBitmap.setPixels(pix, 0, w, 0, 0, w, h);
		return mBitmap;
	}

	/**
	 * create a transition drawable from a bitmap
	 *
	 * @param bitmap bitmap to use for the transition drawable
	 * @return transition drawable
	 */
	public static Drawable createTransitionDrawable(Resources resources, Bitmap bitmap) {
		Drawable layerOne = new ColorDrawable(resources.getColor(android.R.color.transparent));
		BitmapDrawable layerTwo = new BitmapDrawable(resources, bitmap);
		layerTwo.setFilterBitmap(false);
		layerTwo.setDither(false);
		TransitionDrawable result = new TransitionDrawable(new Drawable[]{layerOne, layerTwo});
		result.setCrossFadeEnabled(true);
		result.startTransition(FADE_IN_TIME);
		return result;
	}

	/**
	 * create a blurred drawable le from a bitmap
	 *
	 * @param bitmap bitmap to use for the blurred drawable
	 * @return blurred bitmap drawable
	 */
	public static Drawable createBlurredDrawable(Resources resources, Bitmap bitmap) {
		Bitmap blur = ImageUtils.createBlurredBitmap(bitmap);
		return new BitmapDrawable(resources, blur);
	}

	/**
	 * create small icon for the launcher
	 *
	 * @param bitmap bitmap to create a small shortcut icon
	 * @return shortcut icon bitmap
	 */
	public static Bitmap createShortcutIcon(Context context, Bitmap bitmap) {
		ActivityManager activityManager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
		int iconsize = activityManager.getLauncherLargeIconSize();
		if (bitmap.getWidth() == 0 || bitmap.getHeight() == 0 || iconsize == 0 || (bitmap.getHeight() <= iconsize && bitmap.getWidth() <= iconsize)) {
			return bitmap;
		} else if (bitmap.getHeight() > bitmap.getWidth()) {
			return Bitmap.createScaledBitmap(bitmap, iconsize * bitmap.getWidth() / bitmap.getHeight(), iconsize, false);
		} else {
			return Bitmap.createScaledBitmap(bitmap, iconsize, iconsize * bitmap.getHeight() / bitmap.getWidth(), false);
		}
	}
}