package org.nuclearfog.apollo.ui.fragments.profile;

import android.os.Bundle;
import android.view.ContextMenu;
import android.view.ContextMenu.ContextMenuInfo;
import android.view.Menu;
import android.view.MenuItem;
import android.view.SubMenu;
import android.view.View;
import android.widget.AdapterView.AdapterContextMenuInfo;

import androidx.annotation.NonNull;

import org.nuclearfog.apollo.R;
import org.nuclearfog.apollo.async.AsyncExecutor.AsyncCallback;
import org.nuclearfog.apollo.async.loader.PlaylistSongLoader;
import org.nuclearfog.apollo.model.Song;
import org.nuclearfog.apollo.store.FavoritesStore;
import org.nuclearfog.apollo.ui.adapters.listview.ProfileSongAdapter;
import org.nuclearfog.apollo.ui.adapters.listview.ProfileSongAdapter.DisplaySetting;
import org.nuclearfog.apollo.ui.dialogs.PlaylistDialog;
import org.nuclearfog.apollo.ui.views.dragdrop.DragSortListView.ItemChangeListener;
import org.nuclearfog.apollo.utils.Constants;
import org.nuclearfog.apollo.utils.ContextMenuItems;
import org.nuclearfog.apollo.utils.MusicUtils;
import org.nuclearfog.apollo.utils.NavUtils;

import java.util.List;

/**
 * This class is used to display all of the songs from a particular playlist.
 *
 * @author Andrew Neal (andrewdneal@gmail.com)
 * @author nuclearfog
 */
public class PlaylistSongFragment extends ProfileFragment implements AsyncCallback<List<Song>>, ItemChangeListener {

	/**
	 * Used to keep context menu items from bleeding into other fragments
	 */
	private static final int GROUP_ID = 0x37B5704;

	/**
	 * The adapter for the list
	 */
	private ProfileSongAdapter mAdapter;

	private PlaylistSongLoader mLoader;

	/**
	 * context menu selection
	 */
	private Song mSong;

	/**
	 * The Id of the playlist the songs belong to
	 */
	private long mPlaylistId;

	/**
	 * selected playlist is in queue
	 */
	private boolean queueIsPlaylist = false;


	@Override
	protected void init(Bundle bundle) {
		mLoader = new PlaylistSongLoader(requireContext());
		mAdapter = new ProfileSongAdapter(requireContext(), DisplaySetting.DISPLAY_PLAYLIST_SETTING, true);
		setAdapter(mAdapter);
		// sets empty list text
		setEmptyText(R.string.empty_playlist);
		// Start the loader
		if (bundle != null) {
			mPlaylistId = bundle.getLong(Constants.ID);
			mLoader.execute(mPlaylistId, this);
		}
	}


	@Override
	public void onDestroy() {
		mLoader.cancel();
		super.onDestroy();
	}


	@Override
	protected void onItemClick(View view, int position, long id) {
		MusicUtils.playAllFromUserItemClick(requireActivity(), mAdapter, position);
		// mark playlist as current queue
		queueIsPlaylist = true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onCreateContextMenu(@NonNull ContextMenu menu, @NonNull View v, ContextMenuInfo menuInfo) {
		super.onCreateContextMenu(menu, v, menuInfo);
		if (menuInfo instanceof AdapterContextMenuInfo) {
			// Get the position of the selected item
			AdapterContextMenuInfo info = (AdapterContextMenuInfo) menuInfo;
			// Create a new song
			mSong = mAdapter.getItem(info.position);
			// Play the song
			menu.add(GROUP_ID, ContextMenuItems.PLAY_SELECTION, Menu.NONE, R.string.context_menu_play_selection);
			// Play next
			menu.add(GROUP_ID, ContextMenuItems.PLAY_NEXT, Menu.NONE, R.string.context_menu_play_next);
			// Add the song to the queue
			menu.add(GROUP_ID, ContextMenuItems.ADD_TO_QUEUE, Menu.NONE, R.string.add_to_queue);
			// Add the song to a playlist
			SubMenu subMenu = menu.addSubMenu(GROUP_ID, ContextMenuItems.ADD_TO_PLAYLIST, Menu.NONE, R.string.add_to_playlist);
			MusicUtils.makePlaylistMenu(requireContext(), GROUP_ID, subMenu, true);
			// View more content by the song artist
			menu.add(GROUP_ID, ContextMenuItems.MORE_BY_ARTIST, Menu.NONE, R.string.context_menu_more_by_artist);
			// Make the song a ringtone
			menu.add(GROUP_ID, ContextMenuItems.USE_AS_RINGTONE, Menu.NONE, R.string.context_menu_use_as_ringtone);
			// Remove the song from playlist
			menu.add(GROUP_ID, ContextMenuItems.REMOVE_FROM_PLAYLIST, Menu.NONE, R.string.context_menu_remove_from_playlist);
			// Delete the song
			menu.add(GROUP_ID, ContextMenuItems.DELETE, Menu.NONE, R.string.context_menu_delete);
		} else {
			// remove selection if an error occurs
			mSong = null;
		}
	}


	@Override
	public boolean onContextItemSelected(@NonNull MenuItem item) {
		if (item.getGroupId() == GROUP_ID && mSong != null) {
			switch (item.getItemId()) {
				case ContextMenuItems.PLAY_SELECTION:
					MusicUtils.play(requireActivity(), mSong.getId());
					return true;

				case ContextMenuItems.PLAY_NEXT:
					MusicUtils.playNext(requireActivity(), mSong.getId());
					return true;

				case ContextMenuItems.ADD_TO_QUEUE:
					MusicUtils.addToQueue(requireActivity(), mSong.getId());
					return true;

				case ContextMenuItems.ADD_TO_FAVORITES:
					FavoritesStore.getInstance(requireContext()).addFavorite(mSong);
					return true;

				case ContextMenuItems.NEW_PLAYLIST:
					PlaylistDialog.show(getParentFragmentManager(), PlaylistDialog.CREATE, 0, mSong.getId(), null);
					return true;

				case ContextMenuItems.PLAYLIST_SELECTED:
					if (item.getIntent() != null) {
						long playlistId = item.getIntent().getLongExtra(Constants.PLAYLIST_ID, -1L);
						if (playlistId != -1) {
							MusicUtils.addToPlaylist(requireActivity(), playlistId, mSong.getId());
							// reload if track was added to this playlist
							if (mPlaylistId == playlistId) {
								mLoader.execute(mPlaylistId, this);
							}
						}
					}
					return true;

				case ContextMenuItems.MORE_BY_ARTIST:
					NavUtils.openArtistProfile(requireActivity(), mSong.getArtist());
					return true;

				case ContextMenuItems.USE_AS_RINGTONE:
					MusicUtils.setRingtone(requireActivity(), mSong.getId());
					return true;

				case ContextMenuItems.DELETE:
					MusicUtils.openDeleteDialog(requireActivity(), mSong.getName(), mSong.getId());
					return true;

				case ContextMenuItems.REMOVE_FROM_PLAYLIST:
					if (MusicUtils.removeFromPlaylist(requireActivity(), mSong.getId(), mPlaylistId)) {
						mAdapter.remove(mSong);
					}
					return true;
			}
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onResult(@NonNull List<Song> songs) {
		if (isAdded()) {
			// Start fresh
			mAdapter.clear();
			// Add the data to the adapter
			for (Song song : songs) {
				mAdapter.add(song);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void remove(int index) {
		Song song = mAdapter.getItem(index);
		if (song != null && MusicUtils.removeFromPlaylist(requireActivity(), song.getId(), mPlaylistId)) {
			mAdapter.remove(song);
		} else {
			// if we end here, nothing changed, revert layout changes
			mAdapter.notifyDataSetChanged();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void drop(int from, int to) {
		if (from > 0 && to > 0 && from != to) {
			if (MusicUtils.movePlaylistTrack(requireContext(), mPlaylistId, from, to, ProfileSongAdapter.HEADER_COUNT)) {
				// update adapter
				Song selectedSong = mAdapter.getItem(from);
				mAdapter.remove(selectedSong);
				mAdapter.insert(selectedSong, to);
				// move track item in the current queue
				if (queueIsPlaylist) {
					MusicUtils.moveQueueItem(requireActivity(), from - ProfileSongAdapter.HEADER_COUNT, to - ProfileSongAdapter.HEADER_COUNT);
				}
			}
		}
		mAdapter.notifyDataSetChanged();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void moveToCurrent() {
		Song song = MusicUtils.getCurrentTrack(requireActivity());
		if (song != null) {
			for (int pos = 0; pos < mAdapter.getCount(); pos++) {
				if (mAdapter.getItemId(pos) == song.getId()) {
					scrollTo(pos);
					break;
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void refresh() {
		mLoader.execute(mPlaylistId, this);
	}
}