package org.nuclearfog.apollo.ui.dialogs;

import android.app.Dialog;
import android.content.DialogInterface;
import android.content.DialogInterface.OnClickListener;
import android.os.Bundle;
import android.widget.ArrayAdapter;
import android.widget.ListAdapter;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;

import org.nuclearfog.apollo.R;
import org.nuclearfog.apollo.ui.activities.ProfileActivity;

import java.util.ArrayList;
import java.util.List;

/**
 * Used when the user touches the image in the header in {@link ProfileActivity}
 * . It provides an easy interface for them to choose a new image, use the old
 * image, or web search for one.
 *
 * @author Andrew Neal (andrewdneal@gmail.com)
 * @author nuclearfog
 */
public class PhotoSelectionDialog extends DialogFragment implements OnClickListener {

	private static final String TAG = "PhotoSelectionDialog";

	private static final String KEY_TITLE = "photo_title";
	private static final String KEY_TYPE = "photo_type";

	public static final int ARTIST = 10;
	public static final int ALBUM = 11;
	public static final int OTHER = 0;

	private static final int IDX_NEW = 0;
	private static final int IDX_OLD = 1;
	private static final int IDX_FETCH = 2;
	private static final int IDX_SEARCH = 3;

	private List<String> mChoices = new ArrayList<>(5);


	/**
	 * @param title The dialog title.
	 */
	public static void show(FragmentManager fm, String title, int type) {
		Bundle args = new Bundle();
		PhotoSelectionDialog photoSelectionDialog;
		Fragment dialog = fm.findFragmentByTag(TAG);

		if (dialog instanceof PhotoSelectionDialog) {
			photoSelectionDialog = (PhotoSelectionDialog) dialog;
		} else {
			photoSelectionDialog = new PhotoSelectionDialog();
		}
		args.putInt(KEY_TYPE, type);
		args.putString(KEY_TITLE, title);
		photoSelectionDialog.setArguments(args);
		photoSelectionDialog.show(fm, TAG);
	}

	/**
	 * {@inheritDoc}
	 */
	@NonNull
	@Override
	public Dialog onCreateDialog(@Nullable Bundle savedInstanceState) {
		int type = OTHER;
		String title = "";

		if (getArguments() != null) {
			type = getArguments().getInt(KEY_TYPE, OTHER);
			title = getArguments().getString(KEY_TITLE, "");
		}
		switch (type) {
			case ARTIST:
				// Select a photo from the gallery
				mChoices.add(IDX_NEW, getString(R.string.new_photo));
				// Option to fetch the old album image
				mChoices.add(IDX_OLD, getString(R.string.old_photo));
				// Option to fetch the old artist image
				mChoices.add(IDX_FETCH, getString(R.string.context_menu_fetch_artist_image));
				// web search for the artist name
				mChoices.add(IDX_SEARCH, getString(R.string.web_search));
				break;

			case ALBUM:
				// Select a photo from the gallery
				mChoices.add(IDX_NEW, getString(R.string.new_photo));
				// Option to fetch the old album image
				mChoices.add(IDX_OLD, getString(R.string.old_photo));
				// Option to fetch the album image
				mChoices.add(IDX_FETCH, getString(R.string.context_menu_fetch_album_art));
				// web search for the album name
				mChoices.add(IDX_SEARCH, getString(R.string.web_search));
				break;

			default:
				// Select a photo from the gallery
				mChoices.add(IDX_NEW, getString(R.string.new_photo));
				// Option to use the default image
				mChoices.add(IDX_OLD, getString(R.string.use_default));
				break;
		}
		ListAdapter adapter = new ArrayAdapter<>(requireContext(), android.R.layout.select_dialog_item, mChoices);
		return new AlertDialog.Builder(requireContext()).setTitle(title).setAdapter(adapter, this).create();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onClick(DialogInterface dialog, int which) {
		if (getActivity() instanceof OnOptionSelectedListener) {
			OnOptionSelectedListener listener = (OnOptionSelectedListener) getActivity();
			switch (which) {
				case IDX_NEW:
					listener.selectNewPhoto();
					break;

				case IDX_OLD:
					listener.selectOldPhoto();
					break;

				case IDX_FETCH:
					listener.fetchArtwork();
					break;

				case IDX_SEARCH:
					listener.searchWeb();
					break;
			}
		}
	}

	/**
	 * callback used to send back result to activity
	 */
	public interface OnOptionSelectedListener {

		/**
		 * called to select a new photo local from gallery
		 */
		void selectNewPhoto();

		/**
		 * called to set the previous photo
		 */
		void selectOldPhoto();

		/**
		 * called to select online photos
		 */
		void fetchArtwork();

		/**
		 * called to search online for artist/album
		 */
		void searchWeb();
	}
}