package org.nuclearfog.apollo.ui.adapters.recyclerview;

import android.graphics.PorterDuff;
import android.view.LayoutInflater;
import android.view.ViewGroup;
import android.widget.SeekBar;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

import org.nuclearfog.apollo.R;
import org.nuclearfog.apollo.store.preferences.AppPreferences;
import org.nuclearfog.apollo.ui.adapters.recyclerview.EqualizerAdapter.EqualizerHolder;

import java.text.NumberFormat;

/**
 * Recyclerview Adapter to show equalizer band items
 *
 * @author nuclearfog
 */
public class EqualizerAdapter extends RecyclerView.Adapter<EqualizerHolder> {

	private static final NumberFormat NUMBER_FORMAT = NumberFormat.getInstance();

	private BandLevelChangeListener listener;
	private int[] level, frequency, range;

	private boolean enabled = true;

	/**
	 * @param listener  listener to call if equalizer level changes
	 * @param frequency array of band frequencies (Hz)
	 * @param range     min/max limits of the band
	 */
	public EqualizerAdapter(BandLevelChangeListener listener, int[] frequency, int[] range) {
		this.listener = listener;
		this.frequency = frequency;
		this.range = range;
		level = new int[frequency.length];
	}

	/**
	 * @inheritDoc
	 */
	@NonNull
	@Override
	public EqualizerHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
		final EqualizerHolder holder = new EqualizerHolder(parent);
		holder.slider.setMax((range[1] - range[0]) / 100);
		holder.slider.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
			@Override
			public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
				if (fromUser) {
					int position = holder.getLayoutPosition();
					if (position != RecyclerView.NO_POSITION) {
						// calculate new band level
						level[position] = progress * 100 + range[0];
						holder.level.setText(NUMBER_FORMAT.format(level[position] / 100.0));
						//
						listener.onBandLevelChange(position, level[position]);
					}
				}
			}

			@Override
			public void onStartTrackingTouch(SeekBar seekBar) {
			}

			@Override
			public void onStopTrackingTouch(SeekBar seekBar) {
			}
		});
		return holder;
	}

	/**
	 * @inheritDoc
	 */
	@Override
	public void onBindViewHolder(@NonNull EqualizerHolder holder, int position) {
		// set enabled
		holder.slider.setEnabled(enabled);
		// calculate seekbar position
		holder.slider.setProgress((level[position] - range[0]) / 100);
		// band level
		holder.level.setText(NUMBER_FORMAT.format(level[position] / 100.0));
		// band frequency
		if (frequency[position] >= 1000) {
			holder.frequency.setText(NUMBER_FORMAT.format(frequency[position] / 1000.0));
			holder.frequency.append("K");
		} else {
			holder.frequency.setText(NUMBER_FORMAT.format(frequency[position]));
		}
	}

	/**
	 * @inheritDoc
	 */
	@Override
	public int getItemCount() {
		return Math.min(level.length, frequency.length);
	}

	/**
	 * enable/disable slider
	 *
	 * @param enable true to enable slider
	 */
	public void setEnabled(boolean enable) {
		this.enabled = enable;
		notifyDataSetChanged();
	}

	/**
	 * set band level
	 */
	public void setBands(int[] level) {
		this.level = level;
		notifyDataSetChanged();
	}

	/**
	 *
	 */
	public static class EqualizerHolder extends RecyclerView.ViewHolder {

		final SeekBar slider;
		final TextView level, frequency;

		EqualizerHolder(ViewGroup parent) {
			super(LayoutInflater.from(parent.getContext()).inflate(R.layout.list_item_equalizer_band, parent, false));
			slider = itemView.findViewById(R.id.eq_seekbar);
			level = itemView.findViewById(R.id.eq_level);
			frequency = itemView.findViewById(R.id.eq_freq);

			AppPreferences mPrefs = AppPreferences.getInstance(parent.getContext());
			slider.getProgressDrawable().setColorFilter(mPrefs.getThemeColor(), PorterDuff.Mode.SRC_IN);
			slider.getThumb().setColorFilter(mPrefs.getThemeColor(), PorterDuff.Mode.SRC_IN);
		}
	}

	/**
	 * Adapter item listener
	 */
	public interface BandLevelChangeListener {

		/**
		 * called when a band seekbar changes
		 *
		 * @param pos   adapter position of the band
		 * @param level level value
		 */
		void onBandLevelChange(int pos, int level);
	}
}