package org.nuclearfog.apollo.ui.views;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Rect;
import android.os.Parcel;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewConfiguration;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.viewpager.widget.ViewPager;
import androidx.viewpager.widget.ViewPager.OnPageChangeListener;

import org.nuclearfog.apollo.R;
import org.nuclearfog.apollo.store.preferences.AppPreferences;
import org.nuclearfog.apollo.utils.ApolloUtils;

import java.util.ArrayList;

/**
 * A TitlePageIndicator is a PageIndicator which displays the title of left view
 * (if exist), the title of the current select view (centered) and the title of
 * the right view (if exist). When the user scrolls the ViewPager then titles are
 * also scrolled.
 */
public class TitlePageIndicator extends View implements OnPageChangeListener {
	/**
	 * Percentage indicating what percentage of the screen width away from
	 * center should the underline be fully faded. A value of 0.25 means that
	 * halfway between the center of the screen and an edge.
	 */
	private static final float SELECTION_FADE_PERCENTAGE = 0.25f;

	/**
	 * Percentage indicating what percentage of the screen width away from
	 * center should the selected text bold turn off. A value of 0.05 means
	 * that 10% between the center and an edge.
	 */
	private static final float BOLD_FADE_PERCENTAGE = 0.05f;

	/**
	 * Title text used when no title is provided by the adapter.
	 */
	private static final String EMPTY_TITLE = "";

	/**
	 *
	 */
	private static final int INVALID_POINTER = -1;

	private Paint mPaintText = new Paint();
	private Paint mPaintFooterLine = new Paint();
	private Paint mPaintFooterIndicator = new Paint();
	private Rect mBounds = new Rect();
	private Path mPath = new Path();

	@Nullable
	private ViewPager mViewPager;
	@Nullable
	private OnCenterItemClickListener mCenterItemClickListener;

	private int mCurrentPage = -1;
	private float mPageOffset;
	private int mScrollState;
	private boolean mBoldText;
	private int mColorText;
	private int mColorSelected;

	private float mFooterIndicatorHeight;
	private float mFooterIndicatorUnderlinePadding;
	private float mFooterPadding;
	private float mTitlePadding;
	private float mTopPadding;
	/**
	 * Left and right side padding for not active view titles.
	 */
	private float mClipPadding;
	private float mFooterLine;
	private int mTouchSlop;
	private float mLastMotionX = -1.0f;
	private int mActivePointerId = INVALID_POINTER;
	private boolean mIsDragging;

	/**
	 *
	 */
	public TitlePageIndicator(@NonNull Context context) {
		this(context, null);
	}

	/**
	 *
	 */
	public TitlePageIndicator(@NonNull Context context, @Nullable AttributeSet attrs) {
		this(context, attrs, 0);
	}

	/**
	 *
	 */
	public TitlePageIndicator(@NonNull Context context, @Nullable AttributeSet attrs, int defStyle) {
		super(context, attrs, defStyle);
		//Load defaults from resources
		Resources res = getResources();
		AppPreferences mPref = AppPreferences.getInstance(context);
		mTouchSlop = ViewConfiguration.get(context).getScaledPagingTouchSlop();
		mFooterLine = res.getDimension(R.dimen.default_title_indicator_footer_line_height);
		mFooterIndicatorHeight = res.getDimension(R.dimen.default_title_indicator_footer_indicator_height);
		mFooterIndicatorUnderlinePadding = res.getDimension(R.dimen.default_title_indicator_footer_indicator_underline_padding);
		mFooterPadding = res.getDimension(R.dimen.default_title_indicator_footer_padding);
		mTopPadding = res.getDimension(R.dimen.default_title_indicator_top_padding);
		mTitlePadding = res.getDimension(R.dimen.default_title_indicator_title_padding);
		mClipPadding = res.getDimension(R.dimen.default_title_indicator_clip_padding);
		mColorSelected = res.getColor(R.color.tpi_selected_text_color);
		mColorText = res.getColor(R.color.tpi_unselected_text_color);
		mBoldText = res.getBoolean(R.bool.default_title_indicator_selected_bold);

		mPaintText.setTextSize(res.getDimension(R.dimen.text_size_micro));
		mPaintText.setAntiAlias(true);
		mPaintFooterLine.setStyle(Paint.Style.FILL_AND_STROKE);
		mPaintFooterLine.setStrokeWidth(mFooterLine);
		mPaintFooterLine.setColor(mPref.getThemeColor());
		mPaintFooterIndicator.setStyle(Paint.Style.FILL_AND_STROKE);
		mPaintFooterIndicator.setColor(mPref.getThemeColor());
	}

	/**
	 * @inheritDoc
	 */
	@Override
	protected void onDraw(@NonNull Canvas canvas) {
		super.onDraw(canvas);
		if (mViewPager == null || mViewPager.getAdapter() == null) {
			return;
		}
		int count = getCount();
		// mCurrentPage is -1 on first start and after orientation changed. If so, retrieve the correct index from viewpager.
		if (mCurrentPage == -1 && mViewPager != null) {
			mCurrentPage = mViewPager.getCurrentItem();
		}
		if (ApolloUtils.isLandscape(getContext()) && count > 0 && mCurrentPage == count) {
			mCurrentPage--;
		}
		//Calculate views bounds
		ArrayList<Rect> bounds = calculateAllBounds(mPaintText);
		int boundsSize = bounds.size();
		//Make sure we're on a page that still exists
		if (mCurrentPage >= boundsSize) {
			setCurrentItem(boundsSize - 1);
			return;
		}
		int left = getLeft();
		float leftClip = left + mClipPadding;
		int width = getWidth();
		int height = getHeight();
		int right = left + width;
		float rightClip = right - mClipPadding;

		int page = mCurrentPage;
		float offsetPercent;
		if (mPageOffset <= 0.5) {
			offsetPercent = mPageOffset;
		} else {
			page += 1;
			offsetPercent = 1 - mPageOffset;
		}
		boolean currentSelected = (offsetPercent <= SELECTION_FADE_PERCENTAGE);
		boolean currentBold = (offsetPercent <= BOLD_FADE_PERCENTAGE);
		float selectedPercent = (SELECTION_FADE_PERCENTAGE - offsetPercent) / SELECTION_FADE_PERCENTAGE;

		//Verify if the current view must be clipped to the screen
		Rect curPageBound = bounds.get(mCurrentPage);
		float curPageWidth = curPageBound.right - curPageBound.left;
		if (curPageBound.left < leftClip) {
			//Try to clip to the screen (left side)
			clipViewOnTheLeft(curPageBound, curPageWidth, left);
		}
		if (curPageBound.right > rightClip) {
			//Try to clip to the screen (right side)
			clipViewOnTheRight(curPageBound, curPageWidth, right);
		}
		//Left views starting from the current position
		if (mCurrentPage > 0) {
			for (int i = mCurrentPage - 1; i >= 0; i--) {
				Rect bound = bounds.get(i);
				//Is left side is outside the screen
				if (bound.left < leftClip) {
					int w = bound.right - bound.left;
					//Try to clip to the screen (left side)
					clipViewOnTheLeft(bound, w, left);
					//Except if there's an intersection with the right view
					Rect rightBound = bounds.get(i + 1);
					//Intersection
					if (bound.right + mTitlePadding > rightBound.left) {
						bound.left = (int) (rightBound.left - w - mTitlePadding);
						bound.right = bound.left + w;
					}
				}
			}
		}
		//Right views starting from the current position
		if (mCurrentPage < (count - 1)) {
			for (int i = mCurrentPage + 1; i < count; i++) {
				Rect bound = bounds.get(i);
				//If right side is outside the screen
				if (bound.right > rightClip) {
					int w = bound.right - bound.left;
					//Try to clip to the screen (right side)
					clipViewOnTheRight(bound, w, right);
					//Except if there's an intersection with the left view
					Rect leftBound = bounds.get(i - 1);
					//Intersection
					if (bound.left - mTitlePadding < leftBound.right) {
						bound.left = (int) (leftBound.right + mTitlePadding);
						bound.right = bound.left + w;
					}
				}
			}
		}
		//Now draw views
		int colorTextAlpha = mColorText >>> 24;
		for (int i = 0; i < count; i++) {
			//Get the title
			Rect bound = bounds.get(i);
			//Only if one side is visible
			if ((bound.left > left && bound.left < right) || (bound.right > left && bound.right < right)) {
				boolean currentPage = (i == page);
				CharSequence pageTitle = getTitle(i);

				//Only set bold if we are within bounds
				mPaintText.setFakeBoldText(currentPage && currentBold && mBoldText);

				//Draw text as unselected
				mPaintText.setColor(mColorText);
				if (currentPage && currentSelected) {
					//Fade out/in unselected text as the selected text fades in/out
					mPaintText.setAlpha(colorTextAlpha - Math.round(colorTextAlpha * selectedPercent));
				}

				//Except if there's an intersection with the right view
				if (i < boundsSize - 1) {
					Rect rightBound = bounds.get(i + 1);
					//Intersection
					if (bound.right + mTitlePadding > rightBound.left) {
						int w = bound.right - bound.left;
						bound.left = (int) (rightBound.left - w - mTitlePadding);
						bound.right = bound.left + w;
					}
				}
				canvas.drawText(pageTitle, 0, pageTitle.length(), bound.left, bound.bottom + mTopPadding, mPaintText);

				//If we are within the selected bounds draw the selected text
				if (currentPage && currentSelected) {
					mPaintText.setColor(mColorSelected);
					mPaintText.setAlpha(Math.round((mColorSelected >>> 24) * selectedPercent));
					canvas.drawText(pageTitle, 0, pageTitle.length(), bound.left, bound.bottom + mTopPadding, mPaintText);
				}
			}
		}
		//If we want the line on the top change height to zero and invert the line height to trick the drawing code
		float footerLineHeight = mFooterLine;
		float footerIndicatorLineHeight = mFooterIndicatorHeight;
		//Draw the footer line
		mPath.reset();
		mPath.moveTo(0, height - footerLineHeight / 2f);
		mPath.lineTo(width, height - footerLineHeight / 2f);
		mPath.close();
		canvas.drawPath(mPath, mPaintFooterLine);

		if (currentSelected && page < boundsSize) {
			Rect underlineBounds = bounds.get(page);
			float heightMinusLine = height - footerLineHeight;
			float rightPlusPadding = underlineBounds.right + mFooterIndicatorUnderlinePadding;
			float leftMinusPadding = underlineBounds.left - mFooterIndicatorUnderlinePadding;
			float heightMinusLineMinusIndicator = heightMinusLine - footerIndicatorLineHeight;

			mPath.reset();
			mPath.moveTo(leftMinusPadding, heightMinusLine);
			mPath.lineTo(rightPlusPadding, heightMinusLine);
			mPath.lineTo(rightPlusPadding, heightMinusLineMinusIndicator);
			mPath.lineTo(leftMinusPadding, heightMinusLineMinusIndicator);
			mPath.close();

			mPaintFooterIndicator.setAlpha(Math.round(0xFF * selectedPercent));
			canvas.drawPath(mPath, mPaintFooterIndicator);
			mPaintFooterIndicator.setAlpha(0xFF);
		}
	}

	/**
	 * @inheritDoc
	 */
	@SuppressLint("ClickableViewAccessibility")
	@Override
	public boolean onTouchEvent(android.view.MotionEvent ev) {
		if (super.onTouchEvent(ev)) {
			return true;
		}
		if ((mViewPager == null) || mViewPager.getAdapter() != null && (getCount() == 0)) {
			return false;
		}

		int action = ev.getAction() & MotionEvent.ACTION_MASK;
		switch (action) {
			case MotionEvent.ACTION_DOWN:
				mActivePointerId = ev.getPointerId(0);
				mLastMotionX = ev.getX();
				break;

			case MotionEvent.ACTION_MOVE: {
				int activePointerIndex = ev.findPointerIndex(mActivePointerId);
				float x = ev.getX(activePointerIndex);
				float deltaX = x - mLastMotionX;

				if (!mIsDragging) {
					if (Math.abs(deltaX) > mTouchSlop) {
						mIsDragging = true;
					}
				}

				if (mIsDragging) {
					mLastMotionX = x;
					if (mViewPager.isFakeDragging() || mViewPager.beginFakeDrag()) {
						mViewPager.fakeDragBy(deltaX);
					}
				}

				break;
			}

			case MotionEvent.ACTION_CANCEL:
			case MotionEvent.ACTION_UP:
				if (!mIsDragging) {
					int count = getCount();
					int width = getWidth();
					float halfWidth = width / 2f;
					float sixthWidth = width / 6f;
					float leftThird = halfWidth - sixthWidth;
					float rightThird = halfWidth + sixthWidth;
					float eventX = ev.getX();

					if (eventX < leftThird) {
						if (mCurrentPage > 0) {
							if (action != MotionEvent.ACTION_CANCEL) {
								mViewPager.setCurrentItem(mCurrentPage - 1);
							}
							return true;
						}
					} else if (eventX > rightThird) {
						if (mCurrentPage < count - 1) {
							if (action != MotionEvent.ACTION_CANCEL) {
								mViewPager.setCurrentItem(mCurrentPage + 1);
							}
							return true;
						}
					} else {
						//Middle third
						if (mCenterItemClickListener != null && action != MotionEvent.ACTION_CANCEL) {
							mCenterItemClickListener.onCenterItemClick(mCurrentPage);
						}
					}
				}

				mIsDragging = false;
				mActivePointerId = INVALID_POINTER;
				if (mViewPager.isFakeDragging())
					mViewPager.endFakeDrag();
				break;

			case MotionEvent.ACTION_POINTER_DOWN: {
				int index = ev.getActionIndex();
				mLastMotionX = ev.getX(index);
				mActivePointerId = ev.getPointerId(index);
				break;
			}

			case MotionEvent.ACTION_POINTER_UP:
				int pointerIndex = ev.getActionIndex();
				int pointerId = ev.getPointerId(pointerIndex);
				if (pointerId == mActivePointerId) {
					int newPointerIndex = pointerIndex == 0 ? 1 : 0;
					mActivePointerId = ev.getPointerId(newPointerIndex);
				}
				mLastMotionX = ev.getX(ev.findPointerIndex(mActivePointerId));
				break;
		}

		return true;
	}

	/**
	 * @inheritDoc
	 */
	@Override
	public void onPageScrollStateChanged(int state) {
		mScrollState = state;
	}

	/**
	 * @inheritDoc
	 */
	@Override
	public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
		mCurrentPage = position;
		mPageOffset = positionOffset;
		invalidate();
	}

	/**
	 * @inheritDoc
	 */
	@Override
	public void onPageSelected(int position) {
		if (mScrollState == ViewPager.SCROLL_STATE_IDLE) {
			mCurrentPage = position;
			invalidate();
		}
	}

	/**
	 * @inheritDoc
	 */
	@Override
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		//Measure our width in whatever mode specified
		int measuredWidth = MeasureSpec.getSize(widthMeasureSpec);
		//Determine our height
		float height;
		int heightSpecMode = MeasureSpec.getMode(heightMeasureSpec);
		if (heightSpecMode == MeasureSpec.EXACTLY) {
			//We were told how big to be
			height = MeasureSpec.getSize(heightMeasureSpec);
		} else {
			//Calculate the text bounds
			mBounds.setEmpty();
			mBounds.bottom = (int) (mPaintText.descent() - mPaintText.ascent());
			height = mBounds.bottom - mBounds.top + mFooterLine + mFooterPadding + mTopPadding + mFooterIndicatorHeight;
		}
		int measuredHeight = (int) height;
		setMeasuredDimension(measuredWidth, measuredHeight);
	}

	/**
	 * @inheritDoc
	 */
	@Override
	public void onRestoreInstanceState(Parcelable state) {
		SavedState savedState = (SavedState) state;
		super.onRestoreInstanceState(savedState.getSuperState());
		mCurrentPage = savedState.currentPage;
		requestLayout();
	}

	/**
	 * @inheritDoc
	 */
	@Override
	public Parcelable onSaveInstanceState() {
		Parcelable superState = super.onSaveInstanceState();
		SavedState savedState = new SavedState(superState);
		savedState.currentPage = mCurrentPage;
		return savedState;
	}

	/**
	 *
	 */
	public void setViewPager(@NonNull ViewPager view) {
		if (mViewPager != view) {
			if (mViewPager != null) {
				mViewPager.removeOnPageChangeListener(this);
			}
			mViewPager = view;
			mViewPager.addOnPageChangeListener(this);
			invalidate();
		}
	}

	/**
	 * Set a callback listener for the center item click.
	 *
	 * @param listener Callback instance.
	 */
	public void setOnCenterItemClickListener(OnCenterItemClickListener listener) {
		mCenterItemClickListener = listener;
	}

	/**
	 * Set bounds for the right textView including clip padding.
	 *
	 * @param curViewBound current bounds.
	 * @param curViewWidth width of the view.
	 */
	private void clipViewOnTheRight(Rect curViewBound, float curViewWidth, int right) {
		curViewBound.right = (int) (right - mClipPadding);
		curViewBound.left = (int) (curViewBound.right - curViewWidth);
	}

	/**
	 * Set bounds for the left textView including clip padding.
	 *
	 * @param curViewBound current bounds.
	 * @param curViewWidth width of the view.
	 */
	private void clipViewOnTheLeft(Rect curViewBound, float curViewWidth, int left) {
		curViewBound.left = (int) (left + mClipPadding);
		curViewBound.right = (int) (mClipPadding + curViewWidth);
	}

	/**
	 * Calculate views bounds and scroll them according to the current index
	 */
	private ArrayList<Rect> calculateAllBounds(Paint paint) {
		ArrayList<Rect> list = new ArrayList<>();
		//For each views (If no values then add a fake one)
		if (mViewPager != null && mViewPager.getAdapter() != null) {
			int width = getWidth();
			int halfWidth = width / 2;
			int count = getCount();
			for (int i = 0; i < count; i++) {
				Rect bounds = calcBounds(i, paint);
				int w = bounds.right - bounds.left;
				int h = bounds.bottom - bounds.top;
				bounds.left = (int) (halfWidth - (w / 2f) + ((i - mCurrentPage - mPageOffset) * width));
				bounds.right = bounds.left + w;
				bounds.top = 0;
				bounds.bottom = h;
				list.add(bounds);
			}
		}
		return list;
	}

	/**
	 * Calculate the bounds for a view's title
	 */
	private Rect calcBounds(int index, Paint paint) {
		//Calculate the text bounds
		Rect bounds = new Rect();
		CharSequence title = getTitle(index);
		bounds.right = (int) paint.measureText(title, 0, title.length());
		bounds.bottom = (int) (paint.descent() - paint.ascent());
		return bounds;
	}

	/**
	 *
	 */
	private void setCurrentItem(int item) {
		if (mViewPager != null) {
			mViewPager.setCurrentItem(item);
			mCurrentPage = item;
			invalidate();
		} else {
			throw new IllegalStateException("ViewPager has not been bound.");
		}
	}

	/**
	 *
	 */
	private CharSequence getTitle(int i) {
		CharSequence title = EMPTY_TITLE;
		if (mViewPager != null && mViewPager.getAdapter() != null)
			title = mViewPager.getAdapter().getPageTitle(i);
		return title;
	}

	/**
	 * get page count of the view pager
	 *
	 * @return page count
	 */
	private int getCount() {
		int count = 0;
		if (mViewPager != null && mViewPager.getAdapter() != null) {
			count = mViewPager.getAdapter().getCount();
			if (ApolloUtils.isLandscape(getContext())) {
				count = Math.max(count - 1, 0);
			}
		}
		return count;
	}

	/**
	 * Interface for a callback when the center item has been clicked.
	 */
	public interface OnCenterItemClickListener {
		/**
		 * Callback when the center item has been clicked.
		 *
		 * @param position Position of the current center item.
		 */
		void onCenterItemClick(int position);
	}

	/**
	 *
	 */
	private static class SavedState extends BaseSavedState {

		public static final Parcelable.Creator<SavedState> CREATOR = new Parcelable.Creator<SavedState>() {

			@Override
			public SavedState createFromParcel(Parcel in) {
				return new SavedState(in);
			}

			@Override
			public SavedState[] newArray(int size) {
				return new SavedState[size];
			}
		};

		int currentPage;


		public SavedState(Parcelable superState) {
			super(superState);
		}


		private SavedState(Parcel in) {
			super(in);
			currentPage = in.readInt();
		}


		@Override
		public void writeToParcel(Parcel dest, int flags) {
			super.writeToParcel(dest, flags);
			dest.writeInt(currentPage);
		}
	}
}