package org.nuclearfog.apollo.ui.dialogs;

import android.app.Dialog;
import android.graphics.Color;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.Button;
import android.widget.EditText;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;

import org.nuclearfog.apollo.R;
import org.nuclearfog.apollo.store.preferences.AppPreferences;
import org.nuclearfog.apollo.ui.views.ColorPickerView;
import org.nuclearfog.apollo.utils.NavUtils;

import java.util.Locale;

/**
 * Dialog showing color picker to set the system accent color
 *
 * @author Andrew Neal (andrewdneal@gmail.com)
 * @author nuclearfog
 */
public class ColorSchemeDialog extends DialogFragment implements ColorPickerView.OnColorChangedListener, OnClickListener, TextWatcher {

	private static final String TAG = "ColorSchemeDialog";

	/**
	 * bundle key used to save selected color from previous instance (after screen rotation)
	 */
	private static final String KEY_COLOR = "color_val";

	private ColorPickerView mColorPicker;
	private Button mNewColor;
	private EditText mHexValue;

	private AppPreferences mPreferences;

	/**
	 * show this dialog
	 */
	public static void show(FragmentManager fm) {
		ColorSchemeDialog colorDialog;
		Fragment dialog = fm.findFragmentByTag(TAG);

		if (dialog instanceof ColorSchemeDialog) {
			colorDialog = (ColorSchemeDialog) dialog;
		} else {
			colorDialog = new ColorSchemeDialog();
		}
		colorDialog.show(fm, TAG);
	}

	/**
	 * {@inheritDoc}
	 */
	@NonNull
	@Override
	public Dialog onCreateDialog(@Nullable Bundle savedInstanceState) {
		Dialog dialog = super.onCreateDialog(savedInstanceState);
		if (dialog.getWindow() != null)
			dialog.getWindow().requestFeature(Window.FEATURE_NO_TITLE);
		return dialog;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
		mPreferences = AppPreferences.getInstance(inflater.getContext());
		View mRootView = View.inflate(getContext(), R.layout.dialog_color_scheme, null);
		mColorPicker = mRootView.findViewById(R.id.color_picker_view);
		mNewColor = mRootView.findViewById(R.id.color_scheme_dialog_new_color);
		mHexValue = mRootView.findViewById(R.id.color_scheme_dialog_hex_value);
		Button confirm = mRootView.findViewById(R.id.color_scheme_dialog_apply);
		Button cancel = mRootView.findViewById(R.id.color_scheme_dialog_cancel);
		Button button1 = mRootView.findViewById(R.id.color_scheme_dialog_preset_1);
		Button button2 = mRootView.findViewById(R.id.color_scheme_dialog_preset_2);
		Button button3 = mRootView.findViewById(R.id.color_scheme_dialog_preset_3);
		Button button4 = mRootView.findViewById(R.id.color_scheme_dialog_preset_4);
		Button button5 = mRootView.findViewById(R.id.color_scheme_dialog_preset_5);
		Button button6 = mRootView.findViewById(R.id.color_scheme_dialog_preset_6);
		Button button7 = mRootView.findViewById(R.id.color_scheme_dialog_preset_7);
		Button button8 = mRootView.findViewById(R.id.color_scheme_dialog_preset_8);
		Button mOldColor = mRootView.findViewById(R.id.color_scheme_dialog_old_color);

		int color;
		if (savedInstanceState != null) {
			color = savedInstanceState.getInt(KEY_COLOR);
		} else {
			color = mPreferences.getThemeColor();
		}
		mOldColor.setBackgroundColor(mPreferences.getThemeColor());
		mColorPicker.setColor(color);
		onColorChanged(color);

		mColorPicker.setOnColorChangedListener(this);
		mHexValue.addTextChangedListener(this);
		button1.setOnClickListener(this);
		button2.setOnClickListener(this);
		button3.setOnClickListener(this);
		button4.setOnClickListener(this);
		button5.setOnClickListener(this);
		button6.setOnClickListener(this);
		button7.setOnClickListener(this);
		button8.setOnClickListener(this);
		mOldColor.setOnClickListener(this);
		confirm.setOnClickListener(this);
		cancel.setOnClickListener(this);

		return mRootView;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onSaveInstanceState(@NonNull Bundle outState) {
		outState.putInt(KEY_COLOR, mColorPicker.getColor());
		super.onSaveInstanceState(outState);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onColorChanged(int color) {
		if (mHexValue != null) {
			mHexValue.setText(Integer.toHexString(color).toUpperCase(Locale.getDefault()));
		}
		mNewColor.setBackgroundColor(color);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onClick(View v) {
		if (v.getId() == R.id.color_scheme_dialog_preset_1) {
			mColorPicker.setColorResource(R.color.holo_green);
		} else if (v.getId() == R.id.color_scheme_dialog_preset_2) {
			mColorPicker.setColorResource(R.color.holo_green_light);
		} else if (v.getId() == R.id.color_scheme_dialog_preset_3) {
			mColorPicker.setColorResource(R.color.holo_orange_dark);
		} else if (v.getId() == R.id.color_scheme_dialog_preset_4) {
			mColorPicker.setColorResource(R.color.holo_orange_light);
		} else if (v.getId() == R.id.color_scheme_dialog_preset_5) {
			mColorPicker.setColorResource(R.color.holo_purple);
		} else if (v.getId() == R.id.color_scheme_dialog_preset_6) {
			mColorPicker.setColorResource(R.color.holo_red_light);
		} else if (v.getId() == R.id.color_scheme_dialog_preset_7) {
			mColorPicker.setColorResource(android.R.color.white);
		} else if (v.getId() == R.id.color_scheme_dialog_preset_8) {
			mColorPicker.setColorResource(android.R.color.black);
		} else if (v.getId() == R.id.color_scheme_dialog_old_color) {
			mColorPicker.setColor(mPreferences.getThemeColor());
		} else if (v.getId() == R.id.color_scheme_dialog_cancel) {
			dismiss();
			return;
		} else if (v.getId() == R.id.color_scheme_dialog_apply) {
			mPreferences.setThemeColor(mColorPicker.getColor());
			NavUtils.goHome(getActivity());
			dismiss();
			return;
		}
		onColorChanged(mColorPicker.getColor());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onTextChanged(CharSequence s, int start, int before, int count) {
		try {
			int color = Color.parseColor("#" + mHexValue.getText().toString().toUpperCase(Locale.getDefault()));
			mColorPicker.setColor(color);
			mNewColor.setBackgroundColor(color);
		} catch (IllegalArgumentException e) {
			Log.w(TAG, "color hex format invalid: " + s);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void beforeTextChanged(CharSequence s, int start, int count, int after) {
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void afterTextChanged(Editable s) {
	}
}