/*
 * This file is part of Open Sudoku - an open-source Sudoku game.
 * Copyright (C) 2009-2025 by Open Sudoku authors.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package org.moire.opensudoku.gui.fragments

import android.app.Dialog
import android.os.Bundle
import android.widget.AdapterView
import android.widget.ListView
import androidx.appcompat.app.AlertDialog
import androidx.core.os.bundleOf
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.FragmentActivity
import androidx.fragment.app.setFragmentResult
import org.moire.opensudoku.R
import org.moire.opensudoku.gui.screen.puzzle_list.PuzzleListFilter

class FilterDialogFragment : DialogFragment() {
	override fun onCreateDialog(savedInstanceState: Bundle?): Dialog {
		val checkedItems = booleanArrayOf(
			listFilter.showStateNotStarted,
			listFilter.showStatePlaying,
			listFilter.showStateCompleted
		)

		val dialog = AlertDialog.Builder(requireActivity())
			.setIcon(R.drawable.ic_view)
			.setTitle(R.string.filter_by_game_state)
			.setMultiChoiceItems(R.array.game_states, checkedItems, null)
			.setPositiveButton(android.R.string.ok) { _, _ ->
				listFilter.showStateNotStarted = checkedItems[0]
				listFilter.showStatePlaying = checkedItems[1]
				listFilter.showStateCompleted = checkedItems[2]
				setFragmentResult(requestKey, bundleOf())
			}
			.setNegativeButton(android.R.string.cancel, null)
			.create()

		dialog.listView.onItemClickListener = AdapterView.OnItemClickListener { _, _, which, _ ->
			val listView = dialog.listView
			val isChecked = listView.isItemChecked(which)

			// If the item is now unchecked and it was the last one revert the change by checking it again.
			if (!isChecked && listView.checkedItemCount == 0) {
				listView.setItemChecked(which, true)
			} else {
				checkedItems[which] = isChecked
			}

			updateItemsEnabledState(listView)
		}

		dialog.setOnShowListener {
			updateItemsEnabledState(dialog.listView)
		}

		return dialog
	}

	private fun updateItemsEnabledState(listView: ListView) {
		val isSingleItemSelected = listView.checkedItemCount == 1

		for (i in 0 until listView.count) {
			val view = listView.getChildAt(i - listView.firstVisiblePosition) ?: continue

			// If only one item is selected overall, disable it and enable others otherwise enable all.
			if (isSingleItemSelected) view.isEnabled = !listView.isItemChecked(i)
			else view.isEnabled = true
		}
	}

	companion object {
		lateinit var listFilter: PuzzleListFilter
		private val requestKey: String = this::class.java.declaringClass.simpleName

		fun setListener(parent: FragmentActivity, callback: () -> Unit) {
			parent.supportFragmentManager.setFragmentResultListener(requestKey, parent) { _, _ ->
				callback()
			}
		}
	}
}
