package org.mlm.mages.ui.viewmodel

import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.receiveAsFlow
import kotlinx.coroutines.launch
import org.mlm.mages.MatrixService
import org.mlm.mages.RoomSummary
import org.mlm.mages.matrix.LatestRoomEvent
import org.mlm.mages.matrix.MatrixPort
import org.mlm.mages.matrix.RoomListEntry
import org.mlm.mages.ui.LastMessageType
import org.mlm.mages.ui.RoomListItemUi
import org.mlm.mages.ui.RoomsUiState

class RoomsViewModel(
    private val service: MatrixService
) : BaseViewModel<RoomsUiState>(RoomsUiState(isLoading = true)) {

    // One-time events
    sealed class Event {
        data class OpenRoom(val roomId: String, val name: String) : Event()
        data class ShowError(val message: String) : Event()
    }

    private val _events = Channel<Event>(Channel.BUFFERED)
    val events = _events.receiveAsFlow()

    private var connToken: ULong? = null
    private var roomListToken: ULong? = null
    private var initialized = false
    private var observerJob: Job? = null

    init {
        observerJob = launch {
            while (!service.isLoggedIn()) {
                delay(100)
            }

            service.activeAccount.collect { account ->
                if (account != null) {
                    resetObservers()
                    observeConnection()
                    bootstrapRoomListFromCache()
                    observeRoomList()
                }
            }
        }
    }

    private fun resetObservers() {
        roomListToken?.let {
            runCatching { service.portOrNull?.unobserveRoomList(it) }
        }
        roomListToken = null

        connToken?.let {
            runCatching { service.portOrNull?.stopConnectionObserver(it) }
        }
        connToken = null

        initialized = false
        updateState { RoomsUiState(isLoading = true) }
    }

    //  Public Actions

    fun setSearchQuery(query: String) {
        updateState { copy(roomSearchQuery = query) }
        recomputeGroupedRooms()
    }

    fun toggleUnreadOnly() {
        val next = !currentState.unreadOnly
        updateState { copy(unreadOnly = next) }

        roomListToken?.let { token ->
            runCatching { service.port.roomListSetUnreadOnly(token, next) }
        }

        recomputeGroupedRooms()
    }

    fun openRoom(room: RoomSummary) {
        launch {
            _events.send(Event.OpenRoom(room.id, room.name))
        }
    }

    fun refresh() {
        updateState { copy(isLoading = true) }
        runCatching { service.port.enterForeground() }
    }

    private fun mapRoomSummary(entry: RoomListEntry): RoomSummary {
        return RoomSummary(
            id = entry.roomId,
            name = entry.name,
            avatarUrl = entry.avatarUrl,
            isDm = entry.isDm,
            isEncrypted = entry.isEncrypted
        )
    }

    private fun mapRoomEntryToUi(entry: RoomListEntry): RoomListItemUi {
        val lastEvent = entry.latestEvent
        val lastType = determineMessageType(lastEvent)
        val lastBody = formatBodyForPreview(lastEvent, lastType)

        return RoomListItemUi(
            roomId = entry.roomId,
            name = entry.name,
            avatarUrl = entry.avatarUrl,
            isDm = entry.isDm,
            isEncrypted = entry.isEncrypted,
            unreadCount = entry.notifications.toInt(),
            isFavourite = entry.isFavourite,
            isLowPriority = entry.isLowPriority,
            lastMessageBody = lastBody,
            lastMessageSender = lastEvent?.sender,
            lastMessageType = lastType,
            lastMessageTs = lastEvent?.timestamp
        )
    }

    private fun determineMessageType(event: LatestRoomEvent?): LastMessageType {
        if (event == null) return LastMessageType.Unknown
        if (event.isRedacted) return LastMessageType.Redacted

        val msgtype = event.msgtype
        val evType = event.eventType

        return when {
            msgtype == "m.image"    -> LastMessageType.Image
            msgtype == "m.video"    -> LastMessageType.Video
            msgtype == "m.audio"    -> LastMessageType.Audio
            msgtype == "m.file"     -> LastMessageType.File
            msgtype == "m.sticker"  -> LastMessageType.Sticker
            msgtype == "m.location" -> LastMessageType.Location
            evType  == "m.poll.start"   -> LastMessageType.Poll
            evType  == "m.call.invite"  -> LastMessageType.Call
            event.isEncrypted && event.body == null -> LastMessageType.Encrypted
            else -> LastMessageType.Text
        }
    }

    private fun formatBodyForPreview(event: LatestRoomEvent?, type: LastMessageType): String? {
        if (event == null) return null
        val body = event.body

        if (body != null && body.startsWith("mxc://")) {
            return when (type) {
                LastMessageType.Image -> "Photo"
                LastMessageType.Video -> "Video"
                LastMessageType.Audio -> "Audio"
                LastMessageType.File  -> "File"
                else -> null
            }
        }
        return body
    }

    //  Private Methods

    private fun observeRoomList() {
        if (roomListToken != null) return

        try {
            roomListToken = service.port.observeRoomList(object : MatrixPort.RoomListObserver {
                override fun onReset(items: List<RoomListEntry>) {
                    initialized = true

                    if (items.isEmpty() && currentState.allItems.isNotEmpty() && currentState.offlineBanner != null) {
                        updateState { copy(isLoading = false) }
                        return
                    }

                    items.forEach { maybePrefetchRoomAvatar(it.roomId, it.avatarUrl) }
                    val domainRooms = items.map(::mapRoomSummary)
                    val uiItems     = items.map(::mapRoomEntryToUi)

                    updateState {
                        copy(
                            rooms = domainRooms,
                            unread = items.associate { e -> e.roomId to e.notifications.toInt() },
                            favourites = items.filter { e -> e.isFavourite }.map { e -> e.roomId }.toSet(),
                            lowPriority = items.filter { e -> e.isLowPriority }.map { e -> e.roomId }.toSet(),
                            allItems = uiItems,
                            isLoading = false
                        )
                    }
                    recomputeGroupedRooms()
                }

                override fun onUpdate(item: RoomListEntry) {
                    maybePrefetchRoomAvatar(item.roomId, item.avatarUrl)
                    updateState {
                        val updatedRooms = rooms.map { room ->
                            if (room.id == item.roomId) mapRoomSummary(item) else room
                        }

                        val updatedUiItems = allItems.map { existing ->
                            if (existing.roomId == item.roomId) mapRoomEntryToUi(item) else existing
                        }

                        val updatedUnread = unread.toMutableMap().apply {
                            put(item.roomId, item.notifications.toInt())
                        }

                        val updatedFavourites =
                            if (item.isFavourite) favourites + item.roomId else favourites - item.roomId
                        val updatedLowPriority =
                            if (item.isLowPriority) lowPriority + item.roomId else lowPriority - item.roomId

                        copy(
                            rooms = updatedRooms,
                            unread = updatedUnread,
                            favourites = updatedFavourites,
                            lowPriority = updatedLowPriority,
                            allItems = updatedUiItems
                        )
                    }
                    recomputeGroupedRooms()
                }
            })
        } catch (e: Exception) {
            println("Failed to observe room list: ${e.message}")
        }
    }

    private fun observeConnection() {
        if (connToken != null) return

        try {
            connToken = service.portOrNull?.observeConnection(object : MatrixPort.ConnectionObserver {
                override fun onConnectionChange(state: MatrixPort.ConnectionState) {
                    val banner = when (state) {
                        MatrixPort.ConnectionState.Disconnected -> "No connection"
                        MatrixPort.ConnectionState.Reconnecting -> "Reconnecting..."
                        MatrixPort.ConnectionState.Connecting -> "Connecting..."
                        else -> null
                    }
                    updateState {
                        copy(
                            offlineBanner = banner,
                            isLoading = if (banner != null && allItems.isEmpty()) false else isLoading
                        )
                    }
                }
            })
        } catch (e: Exception) {
            println("Failed to observe connection: ${e.message}")
        }
    }

    private fun recomputeGroupedRooms() {
        val s = currentState
        val query = s.roomSearchQuery.trim()

        var list = s.allItems

        if (query.isNotBlank()) {
            list = list.filter {
                it.name.contains(query, ignoreCase = true) ||
                        it.roomId.contains(query, ignoreCase = true)
            }
        }

        if (s.unreadOnly) {
            list = list.filter { it.unreadCount > 0 }
        }

        val favourites  = list.filter { it.isFavourite }
        val lowPriority = list.filter { it.isLowPriority }
        val normal      = list.filter { !it.isFavourite && !it.isLowPriority }

        updateState {
            copy(
                favouriteItems = favourites,
                normalItems = normal,
                lowPriorityItems = lowPriority
            )
        }
    }

    private fun maybePrefetchRoomAvatar(roomId: String, avatarMxc: String?) {
        if (avatarMxc.isNullOrBlank()) return
        if (currentState.roomAvatarPath.containsKey(roomId)) return

        launch {
            try {
                val path = service.avatars.resolve(avatarMxc, px = 96, crop = true) ?: return@launch
                updateState { copy(roomAvatarPath = roomAvatarPath + (roomId to path)) }
            } catch (_: Exception) {
                // Ignore
            }
        }
    }

    private fun bootstrapRoomListFromCache() {
        viewModelScope.launch {
            if (initialized) return@launch
            if (currentState.allItems.isNotEmpty()) return@launch

            val cached = runCatching { service.port.loadRoomListCache() }
                .getOrElse { emptyList() }

            if (cached.isEmpty()) return@launch
            cached.forEach { maybePrefetchRoomAvatar(it.roomId, it.avatarUrl) }

            val domainRooms = cached.map(::mapRoomSummary)
            val uiItems = cached.map(::mapRoomEntryToUi)

            updateState {
                if (initialized || allItems.isNotEmpty()) this
                else copy(
                    rooms = domainRooms,
                    unread = cached.associate { e -> e.roomId to e.notifications.toInt() },
                    favourites = cached.filter { it.isFavourite }.map { it.roomId }.toSet(),
                    lowPriority = cached.filter { it.isLowPriority }.map { it.roomId }.toSet(),
                    allItems = uiItems
                )
            }

            recomputeGroupedRooms()
        }
    }

    override fun onCleared() {
        super.onCleared()
        observerJob?.cancel()

        roomListToken?.let { token ->
            runCatching { service.portOrNull?.unobserveRoomList(token) }
        }
        connToken?.let { token ->
            runCatching { service.portOrNull?.stopConnectionObserver(token) }
        }
    }
}