export default class HctSolver {
    /** Weights for transforming a set of linear RGB coordinates to Y in XYZ. */
    static Y_FROM_LINRGB: [number, number, number];
    static LINRGB_FROM_SCALED_DISCOUNT: [[number, number, number], [number, number, number], [number, number, number]];
    /** Matrix used when converting from linear RGB to CAM16. */
    static SCALED_DISCOUNT_FROM_LINRGB: [[number, number, number], [number, number, number], [number, number, number]];
    /**  Lookup table for plane in XYZ's Y axis (relative luminance) that corresponds to a given
     * L* in L*a*b*. HCT's T is L*, and XYZ's Y is directly correlated to linear RGB, this table
     * allows us to thus find the intersection between HCT and RGB, giving a solution to the
     * RGB coordinates that correspond to a given set of HCT coordinates.
     */
    static CRITICAL_PLANES: number[];
    /**
     * Sanitizes a degree measure as a floating-point number.
     *
     * @return a degree measure between 0.0 (inclusive) and 360.0 (exclusive).
     */
    static sanitizeDegreesDouble(degrees: number): number;
    /** Equation used in CAM16 conversion that removes the effect of chromatic adaptation. */
    static inverseChromaticAdaptation(adapted: number): number;
    /**
     * Finds a color with the given hue, chroma, and Y.
     *
     * @param hueRadians The desired hue in radians.
     * @param chroma The desired chroma.
     * @param y The desired Y.
     * @return The desired color as a hexadecimal integer, if found; 0 otherwise.
     */
    static findResultByJ(hueRadians: number, chroma: number, y: number): number;
    /**
     * Finds an sRGB color with the given hue, chroma, and L*, if possible.
     *
     * @param hueDegrees The desired hue, in degrees.
     * @param chroma The desired chroma.
     * @param lstar The desired L*.
     * @return A hexadecimal representing the sRGB color. The color has sufficiently close hue,
     * chroma, and L* to the desired values, if possible; otherwise, the hue and L* will be
     * sufficiently close, and chroma will be maximized.
     */
    static solveToInt(hueDegrees: number, chroma: number, lstar: number): number;
    /** Ensure X is between 0 and 100. */
    static isBounded(x: number): boolean;
    /**
     * Returns the nth possible vertex of the polygonal intersection.
     *
     * @param y The Y value of the plane.
     * @param n The zero-based index of the point. 0 <= n <= 11.
     * @return The nth possible vertex of the polygonal intersection of the y plane and the RGB cube
     * in linear RGB coordinates, if it exists. If the possible vertex lies outside of the cube,
     * [-1.0, -1.0, -1.0] is returned.
     */
    static nthVertex(y: number, n: number): [number, number, number];
    static chromaticAdaptation(component: number): number;
    /**
     * Returns the hue of a linear RGB color in CAM16.
     *
     * @param linrgb The linear RGB coordinates of a color.
     * @return The hue of the color in CAM16, in radians.
     */
    static hueOf(linrgb: [number, number, number]): number;
    /**
     * Sanitizes a small enough angle in radians.
     *
     * @param angle An angle in radians; must not deviate too much from 0.
     * @return A coterminal angle between 0 and 2pi.
     */
    static sanitizeRadians(angle: number): number;
    /**
     * Cyclic order is the idea that 330° → 5° → 200° is in order, but, 180° → 270° → 210° is not.
     * Visually, A B and C are angles, and they are in cyclic order if travelling from A to C
     * in a way that increases angle (ex. counter-clockwise if +x axis = 0 degrees and +y = 90)
     * means you must cross B.
     * @param a first angle in possibly cyclic triplet
     * @param b second angle in possibly cyclic triplet
     * @param c third angle in possibly cyclic triplet
     * @return true if B is between A and C
     */
    static areInCyclicOrder(a: number, b: number, c: number): boolean;
    /**
     * Finds the segment containing the desired color.
     *
     * @param y The Y value of the color.
     * @param targetHue The hue of the color.
     * @return A list of two sets of linear RGB coordinates, each corresponding to an endpoint of
     * the segment containing the desired color.
     */
    static bisectToSegment(y: number, targetHue: number): [[number, number, number], [number, number, number]];
    static criticalPlaneBelow(x: number): number;
    static criticalPlaneAbove(x: number): number;
    /**
     * Delinearizes an RGB component, returning a floating-point number.
     *
     * @param rgbComponent 0.0 <= rgb_component <= 100.0, represents linear R/G/B channel
     * @return 0.0 <= output <= 255.0, color channel converted to regular RGB space
     */
    static trueDelinearized(rgbComponent: number): number;
    /**
     * Find an intercept using linear interpolation.
     *
     * @param source The starting number.
     * @param mid The number in the middle.
     * @param target The ending number.
     * @return A number t such that lerp(source, target, t) = mid.
     */
    static intercept(source: number, mid: number, target: number): number;
    /**
     * Linearly interpolate between two points in three dimensions.
     *
     * @param source three dimensions representing the starting point
     * @param t the percentage to travel between source and target, from 0 to 1
     * @param target three dimensions representing the end point
     * @return three dimensions representing the point t percent from source to target.
     */
    static lerpPoint(source: [number, number, number], t: number, target: [number, number, number]): [number, number, number];
    /**
     * Intersects a segment with a plane.
     *
     * @param source The coordinates of point A.
     * @param coordinate The R-, G-, or B-coordinate of the plane.
     * @param target The coordinates of point B.
     * @param axis The axis the plane is perpendicular with. (0: R, 1: G, 2: B)
     * @return The intersection point of the segment AB with the plane R=coordinate, G=coordinate,
     *     or B=coordinate
     */
    static setCoordinate(source: [number, number, number], coordinate: number, target: [number, number, number], axis: 0 | 1 | 2): [number, number, number];
    /**
     * Finds a color with the given Y and hue on the boundary of the cube.
     *
     * @param y The Y value of the color.
     * @param targetHue The hue of the color.
     * @return The desired color, in linear RGB coordinates.
     */
    static bisectToLimit(y: number, targetHue: number): number;
}
//# sourceMappingURL=HctSolver.d.ts.map