"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useScrollContainerLogic = void 0;
var _react = require("react");
var _reactNativeReanimated = require("react-native-reanimated");
var _useDebounce = require("use-debounce");
/**
 * This hook computes the animation logic for the scroll container.
 *
 * @param {UseScrollContainerLogicArgs} args
 * @returns {ReturnType<UseScrollContainerLogicArgs>}
 */
const useScrollContainerLogic = _ref => {
  let {
    scrollRef,
    largeHeaderShown,
    largeHeaderExists,
    disableAutoFixScroll = false,
    adjustmentOffset = 4,
    absoluteHeader = false,
    initialAbsoluteHeaderHeight = 0,
    headerFadeInThreshold = 1,
    inverted,
    onScrollWorklet
  } = _ref;
  const [absoluteHeaderHeight, setAbsoluteHeaderHeight] = (0, _react.useState)(initialAbsoluteHeaderHeight);
  const scrollY = (0, _reactNativeReanimated.useSharedValue)(0);
  const largeHeaderHeight = (0, _reactNativeReanimated.useSharedValue)(0);
  const scrollHandler = (0, _reactNativeReanimated.useAnimatedScrollHandler)(event => {
    if (onScrollWorklet) onScrollWorklet(event);
    scrollY.value = event.contentOffset.y;
  }, [onScrollWorklet]);
  const showNavBar = (0, _reactNativeReanimated.useDerivedValue)(() => {
    if (!largeHeaderExists) return (0, _reactNativeReanimated.withTiming)(scrollY.value <= 0 ? 0 : 1, {
      duration: 250
    });
    if (largeHeaderHeight.value < adjustmentOffset) return 0;
    if (largeHeaderShown) {
      largeHeaderShown.value = (0, _reactNativeReanimated.withTiming)(scrollY.value <= largeHeaderHeight.value * headerFadeInThreshold - adjustmentOffset ? 0 : 1, {
        duration: 250
      });
    }
    return (0, _reactNativeReanimated.withTiming)(scrollY.value <= largeHeaderHeight.value * headerFadeInThreshold - adjustmentOffset ? 0 : 1, {
      duration: 250
    });
  }, [largeHeaderExists]);
  const largeHeaderOpacity = (0, _reactNativeReanimated.useDerivedValue)(() => {
    return (0, _reactNativeReanimated.interpolate)(showNavBar.value, [0, 1], [1, 0]);
  });
  const debouncedFixScroll = (0, _useDebounce.useDebouncedCallback)(() => {
    if (disableAutoFixScroll) return;
    if (largeHeaderHeight.value !== 0 && scrollRef && scrollRef.current) {
      if (scrollY.value >= largeHeaderHeight.value / 2 && scrollY.value < largeHeaderHeight.value) {
        // Scroll to end of large header
        (0, _reactNativeReanimated.runOnUI)(() => {
          'worklet';

          (0, _reactNativeReanimated.scrollTo)(scrollRef, 0, largeHeaderHeight.value, true);
        })();
      } else if (scrollY.value >= 0 && scrollY.value < largeHeaderHeight.value / 2) {
        // Scroll to top
        (0, _reactNativeReanimated.runOnUI)(() => {
          'worklet';

          (0, _reactNativeReanimated.scrollTo)(scrollRef, 0, 0, true);
        })();
      }
    }
  }, 50);
  const onAbsoluteHeaderLayout = (0, _react.useCallback)(e => {
    if (absoluteHeader) {
      setAbsoluteHeaderHeight(e.nativeEvent.layout.height);
    }
  }, [absoluteHeader]);
  const scrollViewAdjustments = (0, _react.useMemo)(() => {
    return {
      scrollIndicatorInsets: {
        top: absoluteHeader && !inverted ? absoluteHeaderHeight : 0,
        bottom: absoluteHeader && inverted ? absoluteHeaderHeight : 0
      },
      contentContainerStyle: {
        paddingTop: absoluteHeader && !inverted ? absoluteHeaderHeight : 0,
        paddingBottom: absoluteHeader && inverted ? absoluteHeaderHeight : 0
      }
    };
  }, [inverted, absoluteHeaderHeight, absoluteHeader]);
  return {
    scrollY,
    showNavBar,
    largeHeaderHeight,
    largeHeaderOpacity,
    scrollHandler,
    debouncedFixScroll,
    absoluteHeaderHeight,
    onAbsoluteHeaderLayout,
    scrollViewAdjustments
  };
};
exports.useScrollContainerLogic = useScrollContainerLogic;
//# sourceMappingURL=useScrollContainerLogic.js.map