import React, { useState, useCallback, useMemo, useRef, forwardRef, useImperativeHandle, } from 'react';
import { View, StyleSheet, Image, Platform, } from 'react-native';
import NativeVideoComponent, {} from './specs/VideoNativeComponent';
import { generateHeaderForNative, getReactTag, resolveAssetSourceForVideo, } from './utils';
import { VideoManager } from './specs/VideoNativeComponent';
const Video = forwardRef(({ source, style, resizeMode, posterResizeMode, poster, fullscreen, drm, textTracks, selectedVideoTrack, selectedAudioTrack, selectedTextTrack, onLoadStart, onLoad, onError, onProgress, onSeek, onEnd, onBuffer, onBandwidthUpdate, onControlsVisibilityChange, onExternalPlaybackChange, onFullscreenPlayerWillPresent, onFullscreenPlayerDidPresent, onFullscreenPlayerWillDismiss, onFullscreenPlayerDidDismiss, onReadyForDisplay, onPlaybackRateChange, onVolumeChange, onAudioBecomingNoisy, onPictureInPictureStatusChanged, onRestoreUserInterfaceForPictureInPictureStop, onReceiveAdEvent, onPlaybackStateChanged, onAudioFocusChanged, onIdle, onTimedMetadata, onAudioTracks, onTextTracks, onTextTrackDataChanged, onVideoTracks, onAspectRatio, ...rest }, ref) => {
    const nativeRef = useRef(null);
    const [showPoster, setShowPoster] = useState(!!poster);
    const [isFullscreen, setIsFullscreen] = useState(fullscreen);
    const [_restoreUserInterfaceForPIPStopCompletionHandler, setRestoreUserInterfaceForPIPStopCompletionHandler,] = useState();
    const hasPoster = !!poster;
    const posterStyle = useMemo(() => ({
        ...StyleSheet.absoluteFillObject,
        resizeMode: posterResizeMode && posterResizeMode !== 'none'
            ? posterResizeMode
            : 'contain',
    }), [posterResizeMode]);
    const src = useMemo(() => {
        if (!source) {
            return undefined;
        }
        const resolvedSource = resolveAssetSourceForVideo(source);
        let uri = resolvedSource.uri || '';
        if (uri && uri.match(/^\//)) {
            uri = `file://${uri}`;
        }
        if (!uri) {
            console.log('Trying to load empty source');
        }
        const isNetwork = !!(uri && uri.match(/^(rtp|rtsp|http|https):/));
        const isAsset = !!(uri &&
            uri.match(/^(assets-library|ipod-library|file|content|ms-appx|ms-appdata):/));
        return {
            uri,
            isNetwork,
            isAsset,
            shouldCache: resolvedSource.shouldCache || false,
            type: resolvedSource.type || '',
            mainVer: resolvedSource.mainVer || 0,
            patchVer: resolvedSource.patchVer || 0,
            requestHeaders: generateHeaderForNative(resolvedSource.headers),
            startPosition: resolvedSource.startPosition ?? -1,
            cropStart: resolvedSource.cropStart || 0,
            cropEnd: resolvedSource.cropEnd,
            metadata: resolvedSource.metadata,
            textTracksAllowChunklessPreparation: resolvedSource.textTracksAllowChunklessPreparation,
        };
    }, [source]);
    const _drm = useMemo(() => {
        if (!drm) {
            return;
        }
        return {
            type: drm.type,
            licenseServer: drm.licenseServer,
            headers: generateHeaderForNative(drm.headers),
            contentId: drm.contentId,
            certificateUrl: drm.certificateUrl,
            base64Certificate: drm.base64Certificate,
            useExternalGetLicense: !!drm.getLicense,
        };
    }, [drm]);
    const _selectedTextTrack = useMemo(() => {
        if (!selectedTextTrack) {
            return;
        }
        const typeOfValueProp = typeof selectedTextTrack.value;
        if (typeOfValueProp !== 'number' &&
            typeOfValueProp !== 'string' &&
            typeOfValueProp !== 'undefined') {
            console.warn('invalid type provided to selectedTextTrack.value: ', typeOfValueProp);
            return;
        }
        return {
            type: selectedTextTrack?.type,
            value: `${selectedTextTrack.value}`,
        };
    }, [selectedTextTrack]);
    const _selectedAudioTrack = useMemo(() => {
        if (!selectedAudioTrack) {
            return;
        }
        const typeOfValueProp = typeof selectedAudioTrack.value;
        if (typeOfValueProp !== 'number' &&
            typeOfValueProp !== 'string' &&
            typeOfValueProp !== 'undefined') {
            console.warn('invalid type provided to selectedAudioTrack.value: ', typeOfValueProp);
            return;
        }
        return {
            type: selectedAudioTrack?.type,
            value: `${selectedAudioTrack.value}`,
        };
    }, [selectedAudioTrack]);
    const _selectedVideoTrack = useMemo(() => {
        if (!selectedVideoTrack) {
            return;
        }
        const typeOfValueProp = typeof selectedVideoTrack.value;
        if (typeOfValueProp !== 'number' &&
            typeOfValueProp !== 'string' &&
            typeOfValueProp !== 'undefined') {
            console.warn('invalid type provided to selectedVideoTrack.value: ', typeOfValueProp);
            return;
        }
        return {
            type: selectedVideoTrack?.type,
            value: `${selectedVideoTrack.value}`,
        };
    }, [selectedVideoTrack]);
    const seek = useCallback(async (time, tolerance) => {
        if (isNaN(time) || time === null) {
            throw new Error("Specified time is not a number: '" + time + "'");
        }
        if (!nativeRef.current) {
            console.warn('Video Component is not mounted');
            return;
        }
        const callSeekFunction = () => {
            VideoManager.seek({
                time,
                tolerance: tolerance || 0,
            }, getReactTag(nativeRef));
        };
        Platform.select({
            ios: callSeekFunction,
            android: callSeekFunction,
            default: () => {
                // TODO: Implement VideoManager.seek for windows
                nativeRef.current?.setNativeProps({ seek: time });
            },
        })();
    }, []);
    const presentFullscreenPlayer = useCallback(() => {
        setIsFullscreen(true);
    }, [setIsFullscreen]);
    const dismissFullscreenPlayer = useCallback(() => {
        setIsFullscreen(false);
    }, [setIsFullscreen]);
    const save = useCallback((options) => {
        // VideoManager.save can be null on android & windows
        return VideoManager.save?.(options, getReactTag(nativeRef));
    }, []);
    const pause = useCallback(() => {
        return VideoManager.setPlayerPauseState(true, getReactTag(nativeRef));
    }, []);
    const resume = useCallback(() => {
        return VideoManager.setPlayerPauseState(false, getReactTag(nativeRef));
    }, []);
    const restoreUserInterfaceForPictureInPictureStopCompleted = useCallback((restored) => {
        setRestoreUserInterfaceForPIPStopCompletionHandler(restored);
    }, [setRestoreUserInterfaceForPIPStopCompletionHandler]);
    const setVolume = useCallback((volume) => {
        return VideoManager.setVolume(volume, getReactTag(nativeRef));
    }, []);
    const getCurrentPosition = useCallback(() => {
        return VideoManager.getCurrentPosition(getReactTag(nativeRef));
    }, []);
    const setFullScreen = useCallback((fullScreen) => {
        return VideoManager.setFullScreen(fullScreen, getReactTag(nativeRef));
    }, []);
    const onVideoLoadStart = useCallback((e) => {
        hasPoster && setShowPoster(true);
        onLoadStart?.(e.nativeEvent);
    }, [hasPoster, onLoadStart]);
    const onVideoLoad = useCallback((e) => {
        if (Platform.OS === 'windows') {
            hasPoster && setShowPoster(false);
        }
        onLoad?.(e.nativeEvent);
    }, [onLoad, hasPoster, setShowPoster]);
    const onVideoError = useCallback((e) => {
        onError?.(e.nativeEvent);
    }, [onError]);
    const onVideoProgress = useCallback((e) => {
        onProgress?.(e.nativeEvent);
    }, [onProgress]);
    const onVideoSeek = useCallback((e) => {
        onSeek?.(e.nativeEvent);
    }, [onSeek]);
    const onVideoPlaybackStateChanged = useCallback((e) => {
        onPlaybackStateChanged?.(e.nativeEvent);
    }, [onPlaybackStateChanged]);
    // android only
    const _onTimedMetadata = useCallback((e) => {
        onTimedMetadata?.(e.nativeEvent);
    }, [onTimedMetadata]);
    const _onAudioTracks = useCallback((e) => {
        onAudioTracks?.(e.nativeEvent);
    }, [onAudioTracks]);
    const _onTextTracks = useCallback((e) => {
        onTextTracks?.(e.nativeEvent);
    }, [onTextTracks]);
    const _onTextTrackDataChanged = useCallback((e) => {
        const { ...eventData } = e.nativeEvent;
        delete eventData.target;
        onTextTrackDataChanged?.(eventData);
    }, [onTextTrackDataChanged]);
    const _onVideoTracks = useCallback((e) => {
        onVideoTracks?.(e.nativeEvent);
    }, [onVideoTracks]);
    const _onPlaybackRateChange = useCallback((e) => {
        onPlaybackRateChange?.(e.nativeEvent);
    }, [onPlaybackRateChange]);
    const _onVolumeChange = useCallback((e) => {
        onVolumeChange?.(e.nativeEvent);
    }, [onVolumeChange]);
    const _onReadyForDisplay = useCallback(() => {
        hasPoster && setShowPoster(false);
        onReadyForDisplay?.();
    }, [setShowPoster, hasPoster, onReadyForDisplay]);
    const _onPictureInPictureStatusChanged = useCallback((e) => {
        onPictureInPictureStatusChanged?.(e.nativeEvent);
    }, [onPictureInPictureStatusChanged]);
    const _onAudioFocusChanged = useCallback((e) => {
        onAudioFocusChanged?.(e.nativeEvent);
    }, [onAudioFocusChanged]);
    const onVideoBuffer = useCallback((e) => {
        onBuffer?.(e.nativeEvent);
    }, [onBuffer]);
    const onVideoExternalPlaybackChange = useCallback((e) => {
        onExternalPlaybackChange?.(e.nativeEvent);
    }, [onExternalPlaybackChange]);
    const _onBandwidthUpdate = useCallback((e) => {
        onBandwidthUpdate?.(e.nativeEvent);
    }, [onBandwidthUpdate]);
    const _onReceiveAdEvent = useCallback((e) => {
        onReceiveAdEvent?.(e.nativeEvent);
    }, [onReceiveAdEvent]);
    const _onVideoAspectRatio = useCallback((e) => {
        onAspectRatio?.(e.nativeEvent);
    }, [onAspectRatio]);
    const _onControlsVisibilityChange = useCallback((e) => {
        onControlsVisibilityChange?.(e.nativeEvent);
    }, [onControlsVisibilityChange]);
    const useExternalGetLicense = drm?.getLicense instanceof Function;
    const onGetLicense = useCallback((event) => {
        if (useExternalGetLicense) {
            const data = event.nativeEvent;
            if (data && data.spcBase64) {
                const getLicenseOverride = drm.getLicense(data.spcBase64, data.contentId, data.licenseUrl, data.loadedLicenseUrl);
                const getLicensePromise = Promise.resolve(getLicenseOverride); // Handles both scenarios, getLicenseOverride being a promise and not.
                getLicensePromise
                    .then((result) => {
                    if (result !== undefined) {
                        nativeRef.current &&
                            VideoManager.setLicenseResult(result, data.loadedLicenseUrl, getReactTag(nativeRef));
                    }
                    else {
                        nativeRef.current &&
                            VideoManager.setLicenseResultError('Empty license result', data.loadedLicenseUrl, getReactTag(nativeRef));
                    }
                })
                    .catch(() => {
                    nativeRef.current &&
                        VideoManager.setLicenseResultError('fetch error', data.loadedLicenseUrl, getReactTag(nativeRef));
                });
            }
            else {
                VideoManager.setLicenseResultError('No spc received', data.loadedLicenseUrl, getReactTag(nativeRef));
            }
        }
    }, [drm, useExternalGetLicense]);
    useImperativeHandle(ref, () => ({
        seek,
        presentFullscreenPlayer,
        dismissFullscreenPlayer,
        save,
        pause,
        resume,
        restoreUserInterfaceForPictureInPictureStopCompleted,
        setVolume,
        getCurrentPosition,
        setFullScreen,
    }), [
        seek,
        presentFullscreenPlayer,
        dismissFullscreenPlayer,
        save,
        pause,
        resume,
        restoreUserInterfaceForPictureInPictureStopCompleted,
        setVolume,
        getCurrentPosition,
        setFullScreen,
    ]);
    return (React.createElement(View, { style: style },
        React.createElement(NativeVideoComponent, { ref: nativeRef, ...rest, src: src, drm: _drm, style: StyleSheet.absoluteFill, resizeMode: resizeMode, fullscreen: isFullscreen, restoreUserInterfaceForPIPStopCompletionHandler: _restoreUserInterfaceForPIPStopCompletionHandler, textTracks: textTracks, selectedTextTrack: _selectedTextTrack, selectedAudioTrack: _selectedAudioTrack, selectedVideoTrack: _selectedVideoTrack, onGetLicense: useExternalGetLicense ? onGetLicense : undefined, onVideoLoad: onLoad || hasPoster
                ? onVideoLoad
                : undefined, onVideoLoadStart: onLoadStart || hasPoster ? onVideoLoadStart : undefined, onVideoError: onError ? onVideoError : undefined, onVideoProgress: onProgress ? onVideoProgress : undefined, onVideoSeek: onSeek ? onVideoSeek : undefined, onVideoEnd: onEnd, onVideoBuffer: onBuffer ? onVideoBuffer : undefined, onVideoPlaybackStateChanged: onPlaybackStateChanged ? onVideoPlaybackStateChanged : undefined, onVideoBandwidthUpdate: onBandwidthUpdate ? _onBandwidthUpdate : undefined, onTimedMetadata: onTimedMetadata ? _onTimedMetadata : undefined, onAudioTracks: onAudioTracks ? _onAudioTracks : undefined, onTextTracks: onTextTracks ? _onTextTracks : undefined, onTextTrackDataChanged: onTextTrackDataChanged ? _onTextTrackDataChanged : undefined, onVideoTracks: onVideoTracks ? _onVideoTracks : undefined, onVideoFullscreenPlayerDidDismiss: onFullscreenPlayerDidDismiss, onVideoFullscreenPlayerDidPresent: onFullscreenPlayerDidPresent, onVideoFullscreenPlayerWillDismiss: onFullscreenPlayerWillDismiss, onVideoFullscreenPlayerWillPresent: onFullscreenPlayerWillPresent, onVideoExternalPlaybackChange: onExternalPlaybackChange ? onVideoExternalPlaybackChange : undefined, onVideoIdle: onIdle, onAudioFocusChanged: onAudioFocusChanged ? _onAudioFocusChanged : undefined, onReadyForDisplay: onReadyForDisplay || hasPoster ? _onReadyForDisplay : undefined, onPlaybackRateChange: onPlaybackRateChange ? _onPlaybackRateChange : undefined, onVolumeChange: onVolumeChange ? _onVolumeChange : undefined, onVideoAudioBecomingNoisy: onAudioBecomingNoisy, onPictureInPictureStatusChanged: onPictureInPictureStatusChanged
                ? _onPictureInPictureStatusChanged
                : undefined, onRestoreUserInterfaceForPictureInPictureStop: onRestoreUserInterfaceForPictureInPictureStop, onVideoAspectRatio: onAspectRatio ? _onVideoAspectRatio : undefined, onReceiveAdEvent: onReceiveAdEvent
                ? _onReceiveAdEvent
                : undefined, onControlsVisibilityChange: onControlsVisibilityChange ? _onControlsVisibilityChange : undefined }),
        hasPoster && showPoster ? (React.createElement(Image, { style: posterStyle, source: { uri: poster } })) : null));
});
Video.displayName = 'Video';
export default Video;
//# sourceMappingURL=Video.js.map