/**
 * A color appearance model, based on CAM16, extended to use L* as the lightness dimension, and
 * coupled to a gamut mapping algorithm. Creates a color system, enables a digital design system.
 */
export declare class Cam {
    static CHROMA_SEARCH_ENDPOINT: number;
    static LIGHTNESS_SEARCH_ENDPOINT: number;
    static DL_MAX: number;
    static DE_MAX: number;
    /** The maximum difference between the requested L* and the L* returned. */
    DL_MAX: number;
    /**
     *  When the delta between the floor & ceiling of a binary search for chroma is less than this,
     * the binary search terminates.
     */
    CHROMA_SEARCH_ENDPOINT: number;
    /**
     * When the delta between the floor & ceiling of a binary search for J, lightness in CAM16,
     * is less than this, the binary search terminates.
     */
    LIGHTNESS_SEARCH_ENDPOINT: number;
    /**
     * sRGB specification has D65 whitepoint - Stokes, Anderson, Chandrasekar, Motta - A Standard
     * Default Color Space for the Internet: sRGB, 1996
     */
    WHITE_POINT_D65: [number, number, number];
    mHue: number;
    mChroma: number;
    mJ: number;
    mQ: number;
    mM: number;
    mS: number;
    mJstar: number;
    mAstar: number;
    mBstar: number;
    constructor(hue: number, chroma: number, j: number, q: number, m: number, s: number, jstar: number, astar: number, bstar: number);
    /** Hue in CAM16 */
    getHue(): number;
    /** Chroma in CAM16 */
    getChroma(): number;
    /** Lightness in CAM16 */
    getJ(): number;
    /** a* coordinate in CAM16-UCS */
    getAstar(): number;
    /** Lightness coordinate in CAM16-UCS */
    getJstar(): number;
    /** b* coordinate in CAM16-UCS */
    getBstar(): number;
    /**
     * Create a CAM from lightness, chroma, and hue coordinates. It is assumed those coordinates
     * were measured in the sRGB standard frame.
     */
    static fromJch(j: number, c: number, h: number): Cam;
    /**
     * Create a CAM from lightness, chroma, and hue coordinates, and also specify the frame in which
     * the color is being viewed.
     */
    static fromJchInFrame(j: number, c: number, h: number): Cam;
    /** Returns perceived color as an ARGB integer, as viewed in standard sRGB frame. */
    viewedInSrgb(): number;
    /** Returns color perceived in a frame as an ARGB integer. */
    viewed(frame: Frame): number;
    /**
     * Distance in CAM16-UCS space between two colors.
     *
     * Much like L*a*b* was designed to measure distance between colors, the CAM16 standard
     * defined a color space called CAM16-UCS to measure distance between CAM16 colors.
     */
    distance(other: Cam): number;
    /**
     * Find J, lightness in CAM16 color space, that creates a color with L* = `lstar` in the L*a*b* color space.
     * Returns null if no J could be found that generated a color with L* `lstar`.
     */
    static findCamByJ(hue: number, chroma: number, lstar: number): Cam | null;
    /**
     * Given a hue & chroma in CAM16, L* in L*a*b*, return an ARGB integer. The chroma of the color
     * returned may, and frequently will, be lower than requested. Assumes the color is viewed in
     * the frame defined by the sRGB standard.
     */
    static getInt(hue: number, chroma: number, lstar: number): number;
    /**
     * Given a hue & chroma in CAM16, L* in L*a*b*, and the frame in which the color will be viewed,
     * return an ARGB integer.
     *
     * The chroma of the color returned may, and frequently will, be lower than requested.
     * This is a fundamental property of color that cannot be worked around by engineering.
     * For example, a red hue, with high chroma, and high L* does not exist: red hues have a maximum chroma below 10
     * in light shades, creating pink.
     */
    static getInt_(hue: number, chroma: number, lstar: number, frame: Frame): number;
    static intFromLstar(lstar: number): number;
    static fromIntInFrame(argb: number, frame: Frame): Cam;
    /**
     * Create a color appearance model from a ARGB integer representing a color. It is assumed the
     * color was viewed in the frame defined in the sRGB standard.
     */
    static fromInt(argb: number): Cam;
}
/**
 * Collection of methods for transforming between color spaces.
 *
 * Methods are named $xFrom$Y. For example, lstarFromInt() returns L* from an ARGB integer.
 *
 * These methods, generally, convert colors between the L*a*b*, XYZ, and sRGB spaces.
 *
 * L*a*b* is a perceptually accurate color space. This is particularly important in the L*
 * dimension: it measures luminance and unlike lightness measures traditionally used in UI work via
 * RGB or HSL, this luminance transitions smoothly, permitting creation of pleasing shades of a
 * color, and more pleasing transitions between colors.
 *
 * XYZ is commonly used as an intermediate color space for converting between one color space to
 * another. For example, to convert RGB to L*a*b*, first RGB is converted to XYZ, then XYZ is
 * converted to L*a*b*.
 *
 * sRGB is a "specification originated from work in 1990s through cooperation by Hewlett-Packard
 * and Microsoft, and it was designed to be a standard definition of RGB for the internet, which it
 * indeed became...The standard is based on a sampling of computer monitors at the time...The whole
 * idea of sRGB is that if everyone assumed that RGB meant the same thing, then the results would be
 * consistent, and reasonably good. It worked." - Fairchild, Color Models and Systems: Handbook of
 * Color Psychology, 2015
 */
export declare class CamUtils {
    /**
     * This is a more precise sRGB to XYZ transformation matrix than traditionally
     * used. It was derived using Schlomer's technique of transforming the xyY
     * primaries to XYZ, then applying a correction to ensure mapping from sRGB
     * 1, 1, 1 to the reference white point, D65.
     */
    static SRGB_TO_XYZ: [[number, number, number], [number, number, number], [number, number, number]];
    /**  Transforms XYZ color space coordinates to 'cone'/'RGB' responses in CAM16.*/
    static XYZ_TO_CAM16RGB: [[number, number, number], [number, number, number], [number, number, number]];
    static XYZ_TO_SRGB: [[number, number, number], [number, number, number], [number, number, number]];
    /** Transforms 'cone'/'RGB' responses in CAM16 to XYZ color space coordinates. */
    static CAM16RGB_TO_XYZ: [[number, number, number], [number, number, number], [number, number, number]];
    /**
     * sRGB specification has D65 whitepoint - Stokes, Anderson, Chandrasekar, Motta - A Standard
     * Default Color Space for the Internet: sRGB, 1996
     */
    static WHITE_POINT_D65: [number, number, number];
    /** Returns L* from L*a*b*, perceptual luminance, from an ARGB integer (ColorInt). */
    static lstarFromInt(argb: number): number;
    static lstarFromY(y: number): number;
    static yFromInt(argb: number): number;
    static xyzFromInt(argb: number): [number, number, number];
    static linearized(rgbComponent: number): number;
    /**
     * Converts an L* value to a Y value.
     *
     * L* in L*a*b* and Y in XYZ measure the same quantity, luminance.
     *
     * L* measures perceptual luminance, a linear scale. Y in XYZ measures relative luminance, a
     * logarithmic scale.
     *
     * @param lstar L* in L*a*b*
     * @return Y in XYZ
     */
    static yFromLstar(lstar: number): number;
    /**
     * Clamps an integer between two integers.
     *
     * @return input when min <= input <= max, and either min or max otherwise.
     */
    static clampInt(min: number, max: number, input: number): number;
    /**
     * Delinearizes an RGB component.
     *
     * @param rgbComponent 0 <= rgb_component <= 100, represents linear R/G/B channel
     * @return 0 <= output <= 255, color channel converted to regular RGB space
     */
    static delinearized(rgbComponent: number): number;
    /** Converts a color from RGB components to ARGB format. */
    static argbFromRgb(red: number, green: number, blue: number): number;
    /** Converts a color from ARGB to XYZ. */
    static argbFromXyz(x: number, y: number, z: number): number;
    /**
     * Convert a color appearance model representation to an ARGB color.
     *
     * Note: the returned color may have a lower chroma than requested. Whether a chroma is
     * available depends on luminance. For example, there's no such thing as a high chroma light
     * red, due to the limitations of our eyes and/or physics. If the requested chroma is
     * unavailable, the highest possible chroma at the requested luminance is returned.
     *
     * @param hue    hue, in degrees, in CAM coordinates
     * @param chroma chroma in CAM coordinates.
     * @param lstar  perceptual luminance, L* in L*a*b*
     */
    static CAMToColor(hue: number, chroma: number, lstar: number): number;
    /**
     * Converts an L* value to an ARGB representation.
     *
     * @param lstar L* in L*a*b*
     * @return ARGB representation of grayscale color with lightness matching L*
     */
    static argbFromLstar(lstar: number): number;
    /** Converts a color from linear RGB components to ARGB format. */
    static argbFromLinrgbComponents(r: number, g: number, b: number): number;
    /**
     * The signum function.
     *
     * @return 1 if num > 0, -1 if num < 0, and 0 if num = 0
     */
    static signum(num: number): 0 | 1 | -1;
    static intFromLstar(lstar: number): number;
}
/**
 * The frame, or viewing conditions, where a color was seen. Used, along with a color, to create a
 * color appearance model representing the color.
 *
 * To convert a traditional color to a color appearance model, it requires knowing what
 * conditions the color was observed in. Our perception of color depends on, for example, the tone
 * of the light illuminating the color, how bright that light was, etc.
 *
 * This class is modelled separately from the color appearance model itself because there are a
 * number of calculations during the color => CAM conversion process that depend only on the viewing
 * conditions. Caching those calculations in a Frame instance saves a significant amount of time.
 */
export declare class Frame {
    mN: number;
    mAw: number;
    mNbb: number;
    mNcb: number;
    mC: number;
    mNc: number;
    mRgbD: [number, number, number];
    mFl: number;
    mFlRoot: number;
    mZ: number;
    constructor(n: number, aw: number, nbb: number, ncb: number, c: number, nc: number, rgbD: [number, number, number], fl: number, fLRoot: number, z: number);
    getRgbD(): [number, number, number];
    getFl(): number;
    getNbb(): number;
    getAw(): number;
    getC(): number;
    getFlRoot(): number;
    getNc(): number;
    getNcb(): number;
    getZ(): number;
    getN(): number;
    static make(whitepoint: [number, number, number], adaptingLuminance: number, backgroundLstar: number, surround: number, discountingIlluminant: boolean): Frame;
    static DEFAULT: Frame;
}
//# sourceMappingURL=Cam.d.ts.map