import { useCallback, useMemo, useState } from 'react';
import { interpolate, runOnUI, scrollTo, useDerivedValue, useSharedValue, withTiming, useAnimatedScrollHandler } from 'react-native-reanimated';
import { useDebouncedCallback } from 'use-debounce';
/**
 * This hook computes the animation logic for the scroll container.
 *
 * @param {UseScrollContainerLogicArgs} args
 * @returns {ReturnType<UseScrollContainerLogicArgs>}
 */
export const useScrollContainerLogic = _ref => {
  let {
    scrollRef,
    largeHeaderShown,
    largeHeaderExists,
    disableAutoFixScroll = false,
    adjustmentOffset = 4,
    absoluteHeader = false,
    initialAbsoluteHeaderHeight = 0,
    headerFadeInThreshold = 1,
    inverted,
    onScrollWorklet
  } = _ref;
  const [absoluteHeaderHeight, setAbsoluteHeaderHeight] = useState(initialAbsoluteHeaderHeight);
  const scrollY = useSharedValue(0);
  const largeHeaderHeight = useSharedValue(0);
  const scrollHandler = useAnimatedScrollHandler(event => {
    if (onScrollWorklet) onScrollWorklet(event);
    scrollY.value = event.contentOffset.y;
  }, [onScrollWorklet]);
  const showNavBar = useDerivedValue(() => {
    if (!largeHeaderExists) return withTiming(scrollY.value <= 0 ? 0 : 1, {
      duration: 250
    });
    if (largeHeaderHeight.value < adjustmentOffset) return 0;
    if (largeHeaderShown) {
      largeHeaderShown.value = withTiming(scrollY.value <= largeHeaderHeight.value * headerFadeInThreshold - adjustmentOffset ? 0 : 1, {
        duration: 250
      });
    }
    return withTiming(scrollY.value <= largeHeaderHeight.value * headerFadeInThreshold - adjustmentOffset ? 0 : 1, {
      duration: 250
    });
  }, [largeHeaderExists]);
  const largeHeaderOpacity = useDerivedValue(() => {
    return interpolate(showNavBar.value, [0, 1], [1, 0]);
  });
  const debouncedFixScroll = useDebouncedCallback(() => {
    if (disableAutoFixScroll) return;
    if (largeHeaderHeight.value !== 0 && scrollRef && scrollRef.current) {
      if (scrollY.value >= largeHeaderHeight.value / 2 && scrollY.value < largeHeaderHeight.value) {
        // Scroll to end of large header
        runOnUI(() => {
          'worklet';

          scrollTo(scrollRef, 0, largeHeaderHeight.value, true);
        })();
      } else if (scrollY.value >= 0 && scrollY.value < largeHeaderHeight.value / 2) {
        // Scroll to top
        runOnUI(() => {
          'worklet';

          scrollTo(scrollRef, 0, 0, true);
        })();
      }
    }
  }, 50);
  const onAbsoluteHeaderLayout = useCallback(e => {
    if (absoluteHeader) {
      setAbsoluteHeaderHeight(e.nativeEvent.layout.height);
    }
  }, [absoluteHeader]);
  const scrollViewAdjustments = useMemo(() => {
    return {
      scrollIndicatorInsets: {
        top: absoluteHeader && !inverted ? absoluteHeaderHeight : 0,
        bottom: absoluteHeader && inverted ? absoluteHeaderHeight : 0
      },
      contentContainerStyle: {
        paddingTop: absoluteHeader && !inverted ? absoluteHeaderHeight : 0,
        paddingBottom: absoluteHeader && inverted ? absoluteHeaderHeight : 0
      }
    };
  }, [inverted, absoluteHeaderHeight, absoluteHeader]);
  return {
    scrollY,
    showNavBar,
    largeHeaderHeight,
    largeHeaderOpacity,
    scrollHandler,
    debouncedFixScroll,
    absoluteHeaderHeight,
    onAbsoluteHeaderLayout,
    scrollViewAdjustments
  };
};
//# sourceMappingURL=useScrollContainerLogic.js.map