"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkExternalConfig = exports.writeConfig = exports.loadConfig = exports.CONFIG_FILE_NAME_JSON = exports.CONFIG_FILE_NAME_JS = exports.CONFIG_FILE_NAME_TS = void 0;
const tslib_1 = require("tslib");
const utils_fs_1 = require("@ionic/utils-fs");
const debug_1 = tslib_1.__importDefault(require("debug"));
const path_1 = require("path");
const colors_1 = tslib_1.__importDefault(require("./colors"));
const errors_1 = require("./errors");
const log_1 = require("./log");
const fn_1 = require("./util/fn");
const js_1 = require("./util/js");
const node_1 = require("./util/node");
const promise_1 = require("./util/promise");
const debug = debug_1.default('capacitor:config');
exports.CONFIG_FILE_NAME_TS = 'capacitor.config.ts';
exports.CONFIG_FILE_NAME_JS = 'capacitor.config.js';
exports.CONFIG_FILE_NAME_JSON = 'capacitor.config.json';
async function loadConfig() {
    var _a, _b, _c, _d, _e;
    const appRootDir = process.cwd();
    const cliRootDir = path_1.dirname(__dirname);
    const conf = await loadExtConfig(appRootDir);
    const appId = (_a = conf.extConfig.appId) !== null && _a !== void 0 ? _a : '';
    const appName = (_b = conf.extConfig.appName) !== null && _b !== void 0 ? _b : '';
    const webDir = (_c = conf.extConfig.webDir) !== null && _c !== void 0 ? _c : 'www';
    const cli = await loadCLIConfig(cliRootDir);
    const config = {
        android: await loadAndroidConfig(appRootDir, conf.extConfig, cli),
        ios: await loadIOSConfig(appRootDir, conf.extConfig),
        web: await loadWebConfig(appRootDir, webDir),
        cli,
        app: {
            rootDir: appRootDir,
            appId,
            appName,
            webDir,
            webDirAbs: path_1.resolve(appRootDir, webDir),
            package: (_d = (await fn_1.tryFn(utils_fs_1.readJSON, path_1.resolve(appRootDir, 'package.json')))) !== null && _d !== void 0 ? _d : {
                name: appName,
                version: '1.0.0',
            },
            ...conf,
            bundledWebRuntime: (_e = conf.extConfig.bundledWebRuntime) !== null && _e !== void 0 ? _e : false,
        },
    };
    debug('config: %O', config);
    return config;
}
exports.loadConfig = loadConfig;
async function writeConfig(extConfig, extConfigFilePath) {
    switch (path_1.extname(extConfigFilePath)) {
        case '.json': {
            await utils_fs_1.writeJSON(extConfigFilePath, extConfig, { spaces: 2 });
            break;
        }
        case '.ts': {
            await utils_fs_1.writeFile(extConfigFilePath, formatConfigTS(extConfig));
            break;
        }
    }
}
exports.writeConfig = writeConfig;
async function loadExtConfigTS(rootDir, extConfigName, extConfigFilePath) {
    var _a, _b;
    try {
        const tsPath = node_1.resolveNode(rootDir, 'typescript');
        if (!tsPath) {
            errors_1.fatal('Could not find installation of TypeScript.\n' +
                `To use ${colors_1.default.strong(extConfigName)} files, you must install TypeScript in your project, e.g. w/ ${colors_1.default.input('npm install -D typescript')}`);
        }
        const ts = require(tsPath); // eslint-disable-line @typescript-eslint/no-var-requires
        const extConfigObject = node_1.requireTS(ts, extConfigFilePath);
        const extConfig = (_a = extConfigObject.default) !== null && _a !== void 0 ? _a : extConfigObject;
        return {
            extConfigType: 'ts',
            extConfigName,
            extConfigFilePath: extConfigFilePath,
            extConfig,
        };
    }
    catch (e) {
        if (!errors_1.isFatal(e)) {
            errors_1.fatal(`Parsing ${colors_1.default.strong(extConfigName)} failed.\n\n${(_b = e.stack) !== null && _b !== void 0 ? _b : e}`);
        }
        throw e;
    }
}
async function loadExtConfigJS(rootDir, extConfigName, extConfigFilePath) {
    var _a;
    try {
        return {
            extConfigType: 'js',
            extConfigName,
            extConfigFilePath: extConfigFilePath,
            extConfig: require(extConfigFilePath),
        };
    }
    catch (e) {
        errors_1.fatal(`Parsing ${colors_1.default.strong(extConfigName)} failed.\n\n${(_a = e.stack) !== null && _a !== void 0 ? _a : e}`);
    }
}
async function loadExtConfig(rootDir) {
    var _a;
    const extConfigFilePathTS = path_1.resolve(rootDir, exports.CONFIG_FILE_NAME_TS);
    if (await utils_fs_1.pathExists(extConfigFilePathTS)) {
        return loadExtConfigTS(rootDir, exports.CONFIG_FILE_NAME_TS, extConfigFilePathTS);
    }
    const extConfigFilePathJS = path_1.resolve(rootDir, exports.CONFIG_FILE_NAME_JS);
    if (await utils_fs_1.pathExists(extConfigFilePathJS)) {
        return loadExtConfigJS(rootDir, exports.CONFIG_FILE_NAME_JS, extConfigFilePathJS);
    }
    const extConfigFilePath = path_1.resolve(rootDir, exports.CONFIG_FILE_NAME_JSON);
    return {
        extConfigType: 'json',
        extConfigName: exports.CONFIG_FILE_NAME_JSON,
        extConfigFilePath: extConfigFilePath,
        extConfig: (_a = (await fn_1.tryFn(utils_fs_1.readJSON, extConfigFilePath))) !== null && _a !== void 0 ? _a : {},
    };
}
async function loadCLIConfig(rootDir) {
    const assetsDir = 'assets';
    const assetsDirAbs = path_1.join(rootDir, assetsDir);
    const iosPlatformTemplateArchive = 'ios-template.tar.gz';
    const iosCordovaPluginsTemplateArchive = 'capacitor-cordova-ios-plugins.tar.gz';
    const androidPlatformTemplateArchive = 'android-template.tar.gz';
    const androidCordovaPluginsTemplateArchive = 'capacitor-cordova-android-plugins.tar.gz';
    return {
        rootDir,
        assetsDir,
        assetsDirAbs,
        assets: {
            ios: {
                platformTemplateArchive: iosPlatformTemplateArchive,
                platformTemplateArchiveAbs: path_1.resolve(assetsDirAbs, iosPlatformTemplateArchive),
                cordovaPluginsTemplateArchive: iosCordovaPluginsTemplateArchive,
                cordovaPluginsTemplateArchiveAbs: path_1.resolve(assetsDirAbs, iosCordovaPluginsTemplateArchive),
            },
            android: {
                platformTemplateArchive: androidPlatformTemplateArchive,
                platformTemplateArchiveAbs: path_1.resolve(assetsDirAbs, androidPlatformTemplateArchive),
                cordovaPluginsTemplateArchive: androidCordovaPluginsTemplateArchive,
                cordovaPluginsTemplateArchiveAbs: path_1.resolve(assetsDirAbs, androidCordovaPluginsTemplateArchive),
            },
        },
        package: await utils_fs_1.readJSON(path_1.resolve(rootDir, 'package.json')),
        os: determineOS(process.platform),
    };
}
async function loadAndroidConfig(rootDir, extConfig, cliConfig) {
    var _a, _b, _c, _d, _e, _f;
    const name = 'android';
    const platformDir = (_b = (_a = extConfig.android) === null || _a === void 0 ? void 0 : _a.path) !== null && _b !== void 0 ? _b : 'android';
    const platformDirAbs = path_1.resolve(rootDir, platformDir);
    const appDir = 'app';
    const srcDir = `${appDir}/src`;
    const srcMainDir = `${srcDir}/main`;
    const assetsDir = `${srcMainDir}/assets`;
    const webDir = `${assetsDir}/public`;
    const resDir = `${srcMainDir}/res`;
    let apkPath = `${appDir}/build/outputs/apk/`;
    let flavorPrefix = '';
    const flavor = ((_c = extConfig.android) === null || _c === void 0 ? void 0 : _c.flavor) || '';
    if ((_d = extConfig.android) === null || _d === void 0 ? void 0 : _d.flavor) {
        apkPath = `${apkPath}/${(_e = extConfig.android) === null || _e === void 0 ? void 0 : _e.flavor}`;
        flavorPrefix = `-${(_f = extConfig.android) === null || _f === void 0 ? void 0 : _f.flavor}`;
    }
    const apkName = `app${flavorPrefix}-debug.apk`;
    const buildOutputDir = `${apkPath}/debug`;
    const cordovaPluginsDir = 'capacitor-cordova-android-plugins';
    const studioPath = promise_1.lazy(() => determineAndroidStudioPath(cliConfig.os));
    return {
        name,
        minVersion: '21',
        studioPath,
        platformDir,
        platformDirAbs,
        cordovaPluginsDir,
        cordovaPluginsDirAbs: path_1.resolve(platformDirAbs, cordovaPluginsDir),
        appDir,
        appDirAbs: path_1.resolve(platformDirAbs, appDir),
        srcDir,
        srcDirAbs: path_1.resolve(platformDirAbs, srcDir),
        srcMainDir,
        srcMainDirAbs: path_1.resolve(platformDirAbs, srcMainDir),
        assetsDir,
        assetsDirAbs: path_1.resolve(platformDirAbs, assetsDir),
        webDir,
        webDirAbs: path_1.resolve(platformDirAbs, webDir),
        resDir,
        resDirAbs: path_1.resolve(platformDirAbs, resDir),
        apkName,
        buildOutputDir,
        buildOutputDirAbs: path_1.resolve(platformDirAbs, buildOutputDir),
        flavor,
    };
}
async function loadIOSConfig(rootDir, extConfig) {
    var _a, _b, _c, _d;
    const name = 'ios';
    const podPath = determineCocoapodPath();
    const platformDir = (_b = (_a = extConfig.ios) === null || _a === void 0 ? void 0 : _a.path) !== null && _b !== void 0 ? _b : 'ios';
    const platformDirAbs = path_1.resolve(rootDir, platformDir);
    const scheme = (_d = (_c = extConfig.ios) === null || _c === void 0 ? void 0 : _c.scheme) !== null && _d !== void 0 ? _d : 'App';
    const nativeProjectDir = 'App';
    const nativeProjectDirAbs = path_1.resolve(platformDirAbs, nativeProjectDir);
    const nativeTargetDir = `${nativeProjectDir}/App`;
    const nativeTargetDirAbs = path_1.resolve(platformDirAbs, nativeTargetDir);
    const nativeXcodeProjDir = `${nativeProjectDir}/App.xcodeproj`;
    const nativeXcodeProjDirAbs = path_1.resolve(platformDirAbs, nativeXcodeProjDir);
    const nativeXcodeWorkspaceDirAbs = promise_1.lazy(() => determineXcodeWorkspaceDirAbs(nativeProjectDirAbs));
    const webDirAbs = promise_1.lazy(() => determineIOSWebDirAbs(nativeProjectDirAbs, nativeTargetDirAbs, nativeXcodeProjDirAbs));
    const cordovaPluginsDir = 'capacitor-cordova-ios-plugins';
    return {
        name,
        minVersion: '12.0',
        platformDir,
        platformDirAbs,
        scheme,
        cordovaPluginsDir,
        cordovaPluginsDirAbs: path_1.resolve(platformDirAbs, cordovaPluginsDir),
        nativeProjectDir,
        nativeProjectDirAbs,
        nativeTargetDir,
        nativeTargetDirAbs,
        nativeXcodeProjDir,
        nativeXcodeProjDirAbs,
        nativeXcodeWorkspaceDir: promise_1.lazy(async () => path_1.relative(platformDirAbs, await nativeXcodeWorkspaceDirAbs)),
        nativeXcodeWorkspaceDirAbs,
        webDir: promise_1.lazy(async () => path_1.relative(platformDirAbs, await webDirAbs)),
        webDirAbs,
        podPath,
    };
}
async function loadWebConfig(rootDir, webDir) {
    const platformDir = webDir;
    const platformDirAbs = path_1.resolve(rootDir, platformDir);
    return {
        name: 'web',
        platformDir,
        platformDirAbs,
    };
}
function determineOS(os) {
    switch (os) {
        case 'darwin':
            return "mac" /* Mac */;
        case 'win32':
            return "windows" /* Windows */;
        case 'linux':
            return "linux" /* Linux */;
    }
    return "unknown" /* Unknown */;
}
async function determineXcodeWorkspaceDirAbs(nativeProjectDirAbs) {
    return path_1.resolve(nativeProjectDirAbs, 'App.xcworkspace');
}
async function determineIOSWebDirAbs(nativeProjectDirAbs, nativeTargetDirAbs, nativeXcodeProjDirAbs) {
    const re = /path\s=\spublic[\s\S]+?sourceTree\s=\s([^;]+)/;
    const pbxprojPath = path_1.resolve(nativeXcodeProjDirAbs, 'project.pbxproj');
    try {
        const pbxproj = await utils_fs_1.readFile(pbxprojPath, { encoding: 'utf8' });
        const m = pbxproj.match(re);
        if (m && m[1] === 'SOURCE_ROOT') {
            log_1.logger.warn(`Using the iOS project root for the ${colors_1.default.strong('public')} directory is deprecated.\n` +
                `Please follow the Upgrade Guide to move ${colors_1.default.strong('public')} inside the iOS target directory: ${colors_1.default.strong('https://capacitorjs.com/docs/updating/3-0#move-public-into-the-ios-target-directory')}`);
            return path_1.resolve(nativeProjectDirAbs, 'public');
        }
    }
    catch (e) {
        // ignore
    }
    return path_1.resolve(nativeTargetDirAbs, 'public');
}
async function determineAndroidStudioPath(os) {
    if (process.env.CAPACITOR_ANDROID_STUDIO_PATH) {
        return process.env.CAPACITOR_ANDROID_STUDIO_PATH;
    }
    switch (os) {
        case "mac" /* Mac */:
            return '/Applications/Android Studio.app';
        case "windows" /* Windows */: {
            const { runCommand } = await Promise.resolve().then(() => tslib_1.__importStar(require('./util/subprocess')));
            let p = 'C:\\Program Files\\Android\\Android Studio\\bin\\studio64.exe';
            try {
                if (!(await utils_fs_1.pathExists(p))) {
                    let commandResult = await runCommand('REG', [
                        'QUERY',
                        'HKEY_LOCAL_MACHINE\\SOFTWARE\\Android Studio',
                        '/v',
                        'Path',
                    ]);
                    commandResult = commandResult.replace(/(\r\n|\n|\r)/gm, '');
                    const i = commandResult.indexOf('REG_SZ');
                    if (i > 0) {
                        p = commandResult.substring(i + 6).trim() + '\\bin\\studio64.exe';
                    }
                }
            }
            catch (e) {
                debug(`Error checking registry for Android Studio path: %O`, e);
                break;
            }
            return p;
        }
        case "linux" /* Linux */:
            return '/usr/local/android-studio/bin/studio.sh';
    }
    return '';
}
function determineCocoapodPath() {
    if (process.env.CAPACITOR_COCOAPODS_PATH) {
        return process.env.CAPACITOR_COCOAPODS_PATH;
    }
    return 'pod';
}
function formatConfigTS(extConfig) {
    // TODO: <reference> tags
    return `import { CapacitorConfig } from '@capacitor/cli';

const config: CapacitorConfig = ${js_1.formatJSObject(extConfig)};

export default config;\n`;
}
function checkExternalConfig(config) {
    var _a, _b;
    if (typeof config.extConfig.hideLogs !== 'undefined' ||
        typeof ((_a = config.extConfig.android) === null || _a === void 0 ? void 0 : _a.hideLogs) !== 'undefined' ||
        typeof ((_b = config.extConfig.ios) === null || _b === void 0 ? void 0 : _b.hideLogs) !== 'undefined') {
        log_1.logger.warn(`The ${colors_1.default.strong('hideLogs')} configuration option has been deprecated. ` +
            `Please update to use ${colors_1.default.strong('loggingBehavior')} instead.`);
    }
}
exports.checkExternalConfig = checkExternalConfig;
