package org.libre.agosto.p2play.viewModels

import android.content.SharedPreferences
import android.util.Log
import androidx.core.content.edit
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.libre.agosto.p2play.AppDatabase
import org.libre.agosto.p2play.Database
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.ajax.Auth
import org.libre.agosto.p2play.domain.entities.AccountModel
import org.libre.agosto.p2play.domain.entities.InstanceModel
import org.libre.agosto.p2play.domain.entities.SessionModel
import org.libre.agosto.p2play.domain.entities.TokenModel

class SessionsViewModel(
    private val db: AppDatabase,
    private val settings: SharedPreferences,
    private val oldDb: Database
) : ViewModel() {
    private val client = Auth()

    private val _isLogged = MutableLiveData<Boolean>()
    val isLogged: LiveData<Boolean> = _isLogged

    private val _isLoading = MutableLiveData<Boolean>()
    val isLoading: LiveData<Boolean> = _isLoading

    private val _instance = MutableLiveData<InstanceModel?>()
    val instance: LiveData<InstanceModel?> = _instance

    private val _session = MutableLiveData<SessionModel?>()
    val session: LiveData<SessionModel?> = _session

    private val _userAccount = MutableLiveData<AccountModel?>()
    val userAccount: LiveData<AccountModel?> = _userAccount

    fun loadInstance(): Job? {
        _isLoading.value = true
        val host = settings.getString("hostP2play", "")!!

        if (host === "") {
            _instance.value = null
            _isLoading.value = false
            return null
        }

        ManagerSingleton.url = host

        return viewModelScope.launch {
            val instanceData = withContext(Dispatchers.IO) {
                db.instances().get(host)
            }

            if (instanceData === null) {
                fetchKeys(host)
            } else {
                _instance.value = instanceData
                _isLoading.value = false
            }
        }
    }

    fun logout() {
        viewModelScope.launch {
            _isLogged.value?.let {
                if (it) {
                    val activeSession = session.value!!
                    withContext(Dispatchers.IO) {
                        client.logout(ManagerSingleton.token.token)
                        activeSession.status = false
                        db.sessions().update(activeSession)
                    }

                    _userAccount.value = null
                    _session.value = null
                    _isLogged.value = false
                    ManagerSingleton.logout()
                }
            }
        }
    }

    fun onUserLogger(token: TokenModel): Job {
        val session = SessionModel(
            null,
            ManagerSingleton.url!!,
            refreshToken = token.refreshToken,
            status = true
        )

        return viewModelScope.launch {
            withContext(Dispatchers.IO) {
                db.sessions().create(session)
                delay(500)
            }
            _session.value = session
        }
    }

    private fun fetchKeys(host: String) {
        viewModelScope.launch {
            val result = withContext(Dispatchers.IO) {
                client.getKeys()
            }

            if (result.clientId !== "") {
                val savedInstance = withContext(Dispatchers.IO) {
                    db.instances().create(
                        InstanceModel(
                            clientId = result.clientId,
                            clientSecret = result.clientSecret,
                            host = host
                        )
                    )
                    db.instances().get(host)
                }
                _instance.value = savedInstance
            }

            _isLoading.value = false
        }
    }

    fun loadSession(): Job? {
        if (ManagerSingleton.url == null) {
            return null
        }
        return viewModelScope.launch {
            val session = withContext(Dispatchers.IO) {
                db.sessions().get(ManagerSingleton.url!!)
            }

            try {
                if (session !== null) {
                    val result = withContext(Dispatchers.IO) {
                        val token = client.refreshToken(
                            session.refreshToken,
                            instance.value!!.clientId,
                            instance.value!!.clientSecret
                        )
                        if (token.status == 1) {
                            session.refreshToken = token.refreshToken
                            db.sessions().update(session)

                            ManagerSingleton.token = token

                            val user = client.me(token.token)
                            ManagerSingleton.user = user

                            val account = client.getMyAccount(token.token)
                            return@withContext account
                        } else {
                            // session.status = false
                            // db.sessions().update(session)

                            return@withContext null
                        }
                    }

                    if (result !== null) {
                        _userAccount.value = result
                        _session.value = session
                        _isLogged.value = true
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace()
                Log.d("Session", "Something goes wrong", e)
            }

            _isLoading.value = false
        }
    }

    suspend fun load() {
        _isLoading.value = true
        migrateSession()?.join()
        loadInstance()?.join()
        _isLoading.value = true
        loadSession()?.join()
        _isLoading.value = false
    }

    private fun migrateSession(): Job? {
        val isMigrated = settings.getBoolean("old-sessions-migrated", false)

        if (isMigrated) {
            return null
        }

        val host = settings.getString("hostP2play", "")
        return viewModelScope.launch {
            val token = withContext(Dispatchers.IO) {
                oldDb.getToken()
            }

            if (host === "" || token.status == -1) {
                return@launch
            }

            val session = SessionModel(
                null,
                host!!,
                refreshToken = token.refreshToken,
                status = true
            )

            withContext(Dispatchers.IO) {
                db.sessions().create(session)
            }

            settings.edit(commit = true) {
                putBoolean("old-sessions-migrated", true)
            }
        }
    }
}
