package org.libre.agosto.p2play.viewModels

import android.content.Intent
import android.util.Log
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.withStyle
import androidx.lifecycle.LiveData
import androidx.lifecycle.MediatorLiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import formatResource
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.libre.agosto.p2play.AccountActivity
import org.libre.agosto.p2play.R
import org.libre.agosto.p2play.ReproductorActivity
import org.libre.agosto.p2play.ajax.Notifications
import org.libre.agosto.p2play.domain.entities.NotificationModel
import org.libre.agosto.p2play.domain.enums.NotificationTypeEnum
import org.libre.agosto.p2play.ui.Routes

class NotificationsViewModel : ViewModel() {
    val client = Notifications()

    private val _notifications = MutableLiveData<List<NotificationModel>>()
    val notifications: LiveData<List<NotificationModel>> = _notifications

    val unreadNotifications = MediatorLiveData<List<NotificationModel>>().apply {
        addSource(_notifications) { list ->
            value = list.filter { !it.read }
        }
    }

    fun loadMore() {
        viewModelScope.launch {
            val result = withContext(Dispatchers.IO) {
                client.get(notifications.value?.size ?: 0)
            }
            val data = if (notifications.value !== null) {
                ArrayList(notifications.value!!)
            } else {
                ArrayList()
            }
            data.addAll(result)
            _notifications.postValue(data)
        }
    }

    fun refresh() {
        this._notifications.value = emptyList()
        this.loadMore()
    }

    fun onMarkAllAsRead() {
        viewModelScope.launch {
            val result = withContext(Dispatchers.IO) {
                client.markAllAsRead()
            }

            Log.d("result", result.toString())
            refresh()
        }
    }

    fun onNotificationClicked(notificationId: Int) {
        viewModelScope.launch {
            val result = withContext(Dispatchers.IO) {
                client.markAsRead(notificationId)
            }

            Log.d("result", result.toString())
            refresh()
        }
    }

    fun getNavNotification(notification: NotificationModel): String? {
        var route: String? = null
        when (notification.type) {
            NotificationTypeEnum.MY_VIDEO_PUBLISHED.ordinal,
            NotificationTypeEnum.MY_VIDEO_IMPORT_SUCCESS.ordinal,
            NotificationTypeEnum.NEW_VIDEO_FROM_SUBSCRIPTION.ordinal -> {
                val video = notification.video!!
                // Go video
                route = Routes.VideoPlayer.nav(video.uuid)
            }
            NotificationTypeEnum.COMMENT_MENTION.ordinal,
            NotificationTypeEnum.NEW_COMMENT_ON_MY_VIDEO.ordinal -> {
                val comment = notification.comment!!

                // Go video comment
                // intent.putExtra("videoId", comment.video.uuid)
                route = Routes.VideoPlayer.nav(comment.video.uuid)
            }
            NotificationTypeEnum.NEW_FOLLOW.ordinal -> {
                val actorFollow = notification.actorFollow!!
                // Go account
                // intent.putExtra("accountId", actorFollow.follower.getAccount())
                route = Routes.Account.nav(actorFollow.follower.getAccount())
            }
            else -> {}
        }

        return route
    }
}
