package org.libre.agosto.p2play.viewModels

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.R
import org.libre.agosto.p2play.ajax.Comments
import org.libre.agosto.p2play.domain.entities.CommentaryModel

class CommentsViewModel (private val videoId: Int): ViewModel() {
    val clientComments = Comments()

    private val _commentaries = MutableLiveData(emptyList<CommentaryModel>())
    val commentaries: LiveData<List<CommentaryModel>> = _commentaries

    private val _replies = MutableLiveData<List<CommentaryModel>>()
    val replies: LiveData<List<CommentaryModel>> = _replies

    private val _toastMessage = MutableLiveData<Int?>()
    val toastMessage: LiveData<Int?> = _toastMessage

    private var threadId: Int? = null

    init {
        viewModelScope.launch {
            getCommentaries()
        }
    }

    fun getThread(threadId: Int) {
        this.threadId = threadId
        viewModelScope.launch {
            _replies.value = withContext(Dispatchers.IO) {
                clientComments.getCommentariesThread(videoId, threadId)
            }
        }
    }

    suspend fun comment(commentary: String): Boolean {
        val result = withContext(Dispatchers.IO) {
            clientComments.makeCommentary(ManagerSingleton.token.token, videoId, commentary)
        }
        if (result) {
            showToast(R.string.makedCommentaryMsg)
            // Refresh comments
            getCommentaries()
        } else {
            showToast(R.string.errorCommentaryMsg)
        }

        return result
    }

    suspend fun replyThread(commentary: String): Boolean {
        val result = withContext(Dispatchers.IO) {
            clientComments.replyThread(ManagerSingleton.token.token, videoId, threadId!!, commentary)
        }
        if (result) {
            showToast(R.string.makedCommentaryMsg)
            // Refresh comments
            getThread(threadId!!)
        } else {
            showToast(R.string.errorCommentaryMsg)
        }

        return result
    }

    private suspend fun getCommentaries() {
        val fetchedCommentaries = withContext(Dispatchers.IO) {
            clientComments.getCommentaries(videoId)
        }
        _commentaries.postValue(fetchedCommentaries)
    }

    private fun showToast(stringResId: Int) {
        _toastMessage.postValue(stringResId)
    }

    class Factory(private val videoId: Int) : ViewModelProvider.Factory {
        @Suppress("UNCHECKED_CAST")
        override fun <T : ViewModel> create(modelClass: Class<T>): T {
            return CommentsViewModel(videoId) as T
        }
    }
}
