package org.libre.agosto.p2play.ui.views.videoView

import android.content.Context
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import androidx.media3.exoplayer.ExoPlayer
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.R
import org.libre.agosto.p2play.ajax.Actions
import org.libre.agosto.p2play.ajax.Comments
import org.libre.agosto.p2play.ajax.Videos
import org.libre.agosto.p2play.domain.entities.CommentaryModel
import org.libre.agosto.p2play.domain.entities.VideoModel
import org.libre.agosto.p2play.singletons.PlaybackSingleton

class VideoViewModel: ViewModel() {
    private lateinit var videoId: String
    
    private val _video = MutableLiveData<VideoModel>()
    val video: LiveData<VideoModel> = _video
    
    private val _relatedVideos = MutableLiveData<List<VideoModel>>()
    val relatedVideos: LiveData<List<VideoModel>> = _relatedVideos

    private val _isSubscribed = MutableLiveData<Boolean>(false)
    val isSubscribed: LiveData<Boolean> = _isSubscribed

    private val _rate = MutableLiveData<String>("")
    val rate: LiveData<String> = _rate

    private val _player = MutableLiveData<ExoPlayer>()
    val player: LiveData<ExoPlayer> = _player

    private val _isFullscreen = MutableLiveData(false)
    val isFullscreen: LiveData<Boolean> = _isFullscreen

    private val _toastMessage = MutableLiveData<Int?>()
    val toastMessage: LiveData<Int?> = _toastMessage

    private val clientVideo = Videos()
    private val clientActions = Actions()

    fun setVideo (videoId: String) {
        this.videoId = videoId

        viewModelScope.launch {
            getVideo()
            delay(1_000)

            if (ManagerSingleton.isLogged()) {
                getRate()
                getSubscription()
            }

        }
    }

    fun rateVideo(rate: String) {
        viewModelScope.launch {
            val result = withContext(Dispatchers.IO) {
                clientActions.rate(ManagerSingleton.token.token, _video.value?.id!!, rate)
            }
            getRate()
            // Show appropriate toast message
            if (result == 1) {
                showToast(R.string.rateMsg)
            } else {
                showToast(R.string.errorMsg)
            }
        }
    }

    fun reportVideo (reason: String) {
        viewModelScope.launch {
            val result = withContext(Dispatchers.IO) {
                clientActions.reportVideo(_video.value?.id!!, reason, ManagerSingleton.token.token)
            }
            if (result) {
                showToast(R.string.reportDialogMsg)
            } else {
                showToast(R.string.errorMsg)
            }
        }
    }

    fun onSubscribe() {
        viewModelScope.launch {
            val result = withContext(Dispatchers.IO) {
                if (_isSubscribed.value == true) {
                    clientActions.unSubscribe(ManagerSingleton.token.token, _video.value?.getChannel()!!)
                } else {
                    clientActions.subscribe(ManagerSingleton.token.token, _video.value?.getChannel()!!)
                }
            }
            if (result == 1) {
                if (_isSubscribed.value == true) {
                    showToast(R.string.unSubscribeMsg)
                } else {
                    showToast(R.string.subscribeMsg)
                }
            } else {
                showToast(R.string.errorMsg)
            }

            getSubscription()
        }
    }

    fun onFullscreenChange() {
        _isFullscreen.value = !_isFullscreen.value!!
    }

    fun makePlayer (context: Context) {
        if (PlaybackSingleton.player == null) {
            _player.value = ExoPlayer.Builder(context).build()
            PlaybackSingleton.player = _player.value
        } else {
            _player.value = PlaybackSingleton.player
        }
    }

    private suspend fun getVideo () {
        val fetchedVideo = withContext(Dispatchers.IO) {
            clientVideo.getVideo(videoId, ManagerSingleton.getToken())
        }
        _video.value = fetchedVideo
        PlaybackSingleton.video = fetchedVideo
//        getRate()
//        getSubscription()
    }

    private suspend fun getRate() {
        _rate.value = withContext(Dispatchers.IO) {
            clientActions.getRate(ManagerSingleton.token.token, _video.value?.id!!)
        }
    }

    private suspend fun getSubscription() {
        _isSubscribed.value = withContext(Dispatchers.IO) {
            clientActions.getSubscription(ManagerSingleton.token.token, _video.value?.getChannel()!!)
        }
    }

    private fun getRelatedVideos() {}

    private fun showToast(stringResId: Int) {
        _toastMessage.postValue(stringResId)
    }

    fun clearToast() {
        _toastMessage.postValue(null)
    }

    fun keepAlive() {
        val currentPos = PlaybackSingleton.player?.contentPosition?.toInt() ?: return
        val currentTime = (currentPos / 1000).toInt()

        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                clientActions.watchingVideo(
                    videoId,
                    ManagerSingleton.token.token,
                    currentTime
                )
            }
        }
    }
}
