package org.libre.agosto.p2play.ui.views.videoView

import android.app.Activity
import android.content.pm.ActivityInfo
import androidx.activity.compose.BackHandler
import androidx.annotation.OptIn
import androidx.appcompat.R
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.aspectRatio
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalView
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.core.view.WindowCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.WindowInsetsControllerCompat
import androidx.media3.common.MediaItem
import androidx.media3.common.util.UnstableApi
import androidx.media3.exoplayer.ExoPlayer
import androidx.navigation.NavController
import kotlinx.coroutines.delay
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.ui.Routes
import org.libre.agosto.p2play.ui.components.atoms.rememberExoPlayer
import org.libre.agosto.p2play.ui.components.atoms.Player
import org.libre.agosto.p2play.ui.components.molecules.VideoDescription
import org.libre.agosto.p2play.ui.components.organisms.VideoComments
import org.libre.agosto.p2play.ui.components.organisms.VideoInfo

@OptIn(UnstableApi::class)
@Composable
fun VideoView(
    navController: NavController,
    videoViewModel: VideoViewModel,
    videoId: String,
    modifier: Modifier = Modifier
) {
    val context = LocalContext.current
    val lazyState = rememberLazyListState()
    val video by videoViewModel.video.observeAsState()
    val isSubscribed by videoViewModel.isSubscribed.observeAsState()
    val rate by videoViewModel.rate.observeAsState()
    val player by videoViewModel.player.observeAsState()
    val paddingMod = remember { Modifier
        .fillMaxWidth()
        .padding(horizontal = 6.dp, vertical = 4.dp) }

    val isFullscreen by videoViewModel.isFullscreen.observeAsState()
    val toastMessage by videoViewModel.toastMessage.observeAsState()
    val view = LocalView.current
    val window = (view.context as Activity).window
    val controller = WindowCompat.getInsetsController(window, view)

    var hasLaunched by rememberSaveable { mutableStateOf(false) }

    toastMessage?.let { stringResId ->
        val message = stringResource(id = stringResId)
        ManagerSingleton.toast(message, context)
        // Clear the toast message after showing it
        videoViewModel.clearToast()
    }

    BackHandler(enabled = isFullscreen == true) {
        videoViewModel.onFullscreenChange()
    }

    LaunchedEffect(isFullscreen) {
        val activity = context as? Activity
        if (isFullscreen == true) {
            // Oculta tanto la barra de estado como la de navegación
            controller.hide(WindowInsetsCompat.Type.systemBars())
            // Permite que aparezcan brevemente si el usuario desliza desde el borde
            controller.systemBarsBehavior =
                WindowInsetsControllerCompat.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
            if (player?.videoFormat!!.width > player?.videoFormat!!.height) {
                activity?.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
            }
        } else {
            controller.show(WindowInsetsCompat.Type.systemBars())
            activity?.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED
        }
    }

    LaunchedEffect(Unit) {
        if (!hasLaunched) {
            videoViewModel.makePlayer(context)
            videoViewModel.setVideo(videoId)
        }
    }

    LaunchedEffect(Unit) {
        if (ManagerSingleton.isLogged()) {
            while (true) {
                if (player?.isPlaying == true) {
                    videoViewModel.keepAlive()
                }
                delay(10_000)
            }
        }
    }

    LaunchedEffect(video) {
        if (!hasLaunched) {
            video?.let { v ->
                val isResume = navController.currentBackStackEntry?.arguments?.getBoolean("resume")
                if (isResume != true) {
                    player?.apply {
                        stop()
                        clearMediaItems()
                        setMediaItem(MediaItem.fromUri(v.streamingData?.playlistUrl!!))
                        prepare()
                        playWhenReady = true
                    }
                }

                hasLaunched = true
            }
        }
    }

    LazyColumn(modifier, lazyState) {
        item(key = "video-player") {
            val playerMod = if (isFullscreen == true) { Modifier.fillParentMaxSize() } else {
                Modifier
                    .fillMaxWidth()
                    .aspectRatio((16f / 9f))
            }
            Player({ player }, playerMod) {
                videoViewModel.onFullscreenChange()
            }
        }
        isFullscreen?.let {
            if (!it) {
                item(key = "info") {
                    Box(paddingMod) {
                        video?.let { v ->
                            VideoInfo(
                                v,
                                isSubscribed == true,
                                rate == "like",
                                rate == "dislike",
                                { r ->
                                    videoViewModel.rateVideo(r)
                                },
                                {
                                    videoViewModel.onSubscribe()
                                },
                                { r ->
                                    videoViewModel.reportVideo(r)
                                },
                                {
                                    navController.navigate(Routes.Channel.nav(video!!.getChannel()))
                                }
                            )
                        }
                    }
                }
                item(key = "description") {
                    Box(paddingMod) {
                        video?.let {
                            VideoDescription(it.shortDescription, it.description, {})
                        }
                    }
                }
                item(key = "comments") {
                    Box(paddingMod) {
                        if (video != null) {
                            VideoComments(video?.id!!, video?.commentsEnabled!!)
                        }
                    }
                }
                item(key = "related-videos") {
                    Box(paddingMod) {
                        // TODO: Add related videos
                    }
                }
            }
        }
    }
}
