package org.libre.agosto.p2play.ui.views.channelView

import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Coffee
import androidx.compose.material3.AssistChip
import androidx.compose.material3.Card
import androidx.compose.material3.Icon
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.painter.ColorPainter
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.navigation.NavController
import coil3.compose.AsyncImage
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.R
import org.libre.agosto.p2play.domain.entities.ChannelModel
import org.libre.agosto.p2play.ui.Routes
import org.libre.agosto.p2play.ui.components.atoms.BoldText
import org.libre.agosto.p2play.ui.components.atoms.SupportText
import org.libre.agosto.p2play.ui.components.atoms.UserImg
import org.libre.agosto.p2play.ui.components.molecules.ChannelStats
import org.libre.agosto.p2play.ui.components.molecules.VideoAccount
import org.libre.agosto.p2play.ui.lists.VideoList

@Composable
fun ChannelView (navController: NavController, channelViewModel: ChannelViewModel, channelId: String) {
    val context = LocalContext.current
    val channel by channelViewModel.channel.observeAsState(initial = ChannelModel())
    val videos by channelViewModel.videos.observeAsState(arrayListOf())
    val isSubscribed by channelViewModel.isSubscribed.observeAsState(false)
    val toastMessage by channelViewModel.toastMessage.observeAsState()

    LaunchedEffect(Unit) {
        channelViewModel.getChannel(channelId)
    }

    toastMessage?.let { stringResId ->
        val message = stringResource(id = stringResId)
        ManagerSingleton.toast(message, context)
        // Clear the toast message after showing it
        // videoViewModel.clearToast()
    }

    VideoList(
        videos,
        Modifier.padding(4.dp),
        header = {
            Surface {
                Column(Modifier.fillMaxWidth()) {
                    val bannerMod = Modifier
                        .fillMaxWidth()
                        .clip(RoundedCornerShape(15))
                        .height(70.dp)
                    if (channel.banner != "") {
                        AsyncImage(
                            "https://${ManagerSingleton.url}${channel.banner}",
                            "",
                            bannerMod,
                            contentScale = ContentScale.Crop
                        )
                    } else {
                        Image(ColorPainter(Color.Gray), "", bannerMod)
                    }
                    Spacer(Modifier.height(12.dp))
                    VideoAccount(channel.name, channel.getFullName(), channel.channelImg, isSubscribed) {
                        channelViewModel.onSubscribe()
                    }
                    Spacer(Modifier.height(10.dp))
                    ChannelStats(channel.followers)
                    Spacer(Modifier.height(4.dp))
                    if (channel.description != "") {
                        Card(Modifier.fillMaxWidth()) {
                            Column(Modifier.padding(6.dp)) {
                                BoldText(stringResource(R.string.descriptionTxt))
                                SupportText(channel.description)
                            }
                        }
                    }
                    Row() {
                        AssistChip(
                            onClick = { navController.navigate(Routes.Account.nav(channel.account!!.getAccount())) },
                            label = { Text("By ${channel.account?.name}") },
                            leadingIcon = {
                                UserImg(Modifier.size(24.dp), channel.account?.avatars?.getOrNull(0)?.path)
                            }
                        )
                        Spacer(Modifier.width(6.dp))
                        if(channel.support != "") {
                            AssistChip(
                                onClick = {  },
                                label = { Text("Support") },
                                leadingIcon = {
                                    Icon(Icons.Default.Coffee, "")
                                }
                            )
                        }
                    }
                    Spacer(Modifier.height(6.dp))
                }
            }
        }
    ) {
        navController.navigate(Routes.VideoPlayer.nav(it)) {
            launchSingleTop = true
        }
    }

}
