package org.libre.agosto.p2play.ui.views.accountView

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.grid.GridCells
import androidx.compose.foundation.lazy.grid.GridItemSpan
import androidx.compose.foundation.lazy.grid.LazyGridItemSpanScope
import androidx.compose.foundation.lazy.grid.LazyVerticalGrid
import androidx.compose.foundation.lazy.grid.items
import androidx.compose.foundation.lazy.grid.rememberLazyGridState
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.Card
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.PrimaryTabRow
import androidx.compose.material3.Surface
import androidx.compose.material3.Tab
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.navigation.NavController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.R
import org.libre.agosto.p2play.domain.entities.AccountModel
import org.libre.agosto.p2play.ui.Routes
import org.libre.agosto.p2play.ui.components.atoms.BoldText
import org.libre.agosto.p2play.ui.components.atoms.SupportText
import org.libre.agosto.p2play.ui.components.atoms.UserImg
import org.libre.agosto.p2play.ui.components.molecules.VideoAccount
import org.libre.agosto.p2play.ui.components.organisms.VideoItem
import org.libre.agosto.p2play.ui.lists.VideoList

@Composable
fun AccountView(navController: NavController, accountViewModel: AccountViewModel, accountId: String) {
    val context = LocalContext.current
    val account by accountViewModel.account.observeAsState(AccountModel("", "", "", arrayListOf(), "", ""))
    val videos by accountViewModel.videos.observeAsState(arrayListOf())
    val channels by accountViewModel.channels.observeAsState(arrayListOf())
    val toastMessage by accountViewModel.toastMessage.observeAsState()

    var tabIndex by rememberSaveable { mutableIntStateOf(0) }
    val lazyState = rememberLazyGridState()
    // var spanSize = 2
    val maxSpan: (LazyGridItemSpanScope) -> GridItemSpan = {
        // spanSize = it.maxCurrentLineSpan + 1
        GridItemSpan(it.maxCurrentLineSpan)
    }

    toastMessage?.let { stringResId ->
        val message = stringResource(id = stringResId)
        ManagerSingleton.toast(message, context)
        // Clear the toast message after showing it
        // videoViewModel.clearToast()
    }

    LaunchedEffect(Unit) {
        accountViewModel.getAccount(accountId)
    }

    LazyVerticalGrid(
        GridCells.Adaptive(300.dp),
        state = lazyState,
        modifier = Modifier
            .fillMaxWidth(),
        verticalArrangement = Arrangement.spacedBy(16.dp),
        horizontalArrangement = Arrangement.spacedBy(16.dp)
    ) {
        item ("header", span = maxSpan) {
            Surface() {
                Column(Modifier.fillMaxWidth()) {
                    Column(
                        Modifier.fillMaxWidth(),
                        horizontalAlignment = Alignment.CenterHorizontally
                    ) {
                        UserImg(
                            Modifier.padding(6.dp).size(60.dp),
                            account.avatars.getOrNull(account.avatars.size - 1)?.path
                        )
                        BoldText(account.displayName)
                        SupportText(account.getAccount())
                    }
                    Spacer(Modifier.height(4.dp))
                    Card(Modifier.fillMaxWidth()) {
                        Column(Modifier.padding(6.dp)) {
                            BoldText(stringResource(R.string.descriptionTxt))
                            SupportText(account.description)
                        }
                    }
                    PrimaryTabRow(selectedTabIndex = tabIndex, modifier = Modifier.padding(1.dp)) {
                        Tab(
                            selected = navController.currentBackStackEntry?.id == "videos",
                            onClick = {
                                tabIndex = 0
                            },
                            text = {
                                Text(
                                    text = "Videos",
                                    maxLines = 2,
                                    overflow = TextOverflow.Ellipsis
                                )
                            }
                        )
                        Tab(
                            selected = false,
                            onClick = {
                                tabIndex = 1
                            },
                            text = {
                                Text(
                                    text = "Channels",
                                    maxLines = 2,
                                    overflow = TextOverflow.Ellipsis
                                )
                            }
                        )
//                        Tab(
//                            selected = false,
//                            onClick = {
//                                tabIndex = 2
//                            },
//                            text = {
//                                Text(
//                                    text = "Playlists",
//                                    maxLines = 2,
//                                    overflow = TextOverflow.Ellipsis
//                                )
//                            }
//                        )
                    }
                }
            }
        }

        when(tabIndex) {
            0 -> {
                items(videos, { it.id }) {
                    VideoItem(
                        it,
                        { navController.navigate(Routes.Channel.nav(it)) }
                    ) { videoId ->
                        navController.navigate(Routes.VideoPlayer.nav(videoId))
                    }
                }
            }
            1 -> {
                items(channels, { it.id }) {
                    VideoAccount(
                        it.name,
                        it.getFullName(),
                        it.channelImg,
                        it.subscribed,
                        { navController.navigate(Routes.Channel.nav(it.getFullName())) }
                    ) {
                        accountViewModel.onSubscribe(it)
                    }

                }
            }
            else -> {}
        }
    }
}
