package org.libre.agosto.p2play.ui.views

import android.annotation.SuppressLint
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.compose.LocalLifecycleOwner
import androidx.navigation.NavController
import java.util.ArrayList
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.ui.Routes
import org.libre.agosto.p2play.ui.components.molecules.LoginBanner
import org.libre.agosto.p2play.ui.lists.SubscriptionIconList
import org.libre.agosto.p2play.ui.lists.VideoList
import org.libre.agosto.p2play.viewModels.SessionsViewModel
import org.libre.agosto.p2play.viewModels.SubscriptionsViewModel

@SuppressLint("MutableCollectionMutableState")
@Composable
fun SubscriptionsView(
    navController: NavController,
    subscriptionsViewModel: SubscriptionsViewModel,
    sessionsViewModel: SessionsViewModel,
    modifier: Modifier = Modifier
) {
    val lifecycleOwner = LocalLifecycleOwner.current
    val videos by subscriptionsViewModel.videos.observeAsState(initial = listOf())
    val subscriptions by subscriptionsViewModel.subscriptions.observeAsState(initial = listOf())
    val isLoading: Boolean by subscriptionsViewModel.isLoading.observeAsState(initial = false)

    val isLogged by sessionsViewModel.isLogged.observeAsState(false)

    DisposableEffect(lifecycleOwner) {
        val observer = LifecycleEventObserver { _, event ->
            if (event == Lifecycle.Event.ON_RESUME) {
                sessionsViewModel.loadSession()
            }
        }

        lifecycleOwner.lifecycle.addObserver(observer)

        onDispose {
            lifecycleOwner.lifecycle.removeObserver(observer)
        }
    }

    LaunchedEffect(isLogged) {
        if (isLogged) {
            if (videos.isEmpty()) {
                subscriptionsViewModel.loadVideos()
            }

            if (subscriptions.isEmpty()) {
                subscriptionsViewModel.loadSubscriptions()
            }
        }
    }

    LaunchedEffect(Unit) {
        if (!isLogged) {
            return@LaunchedEffect
        }
        if (videos.isEmpty()) {
            subscriptionsViewModel.loadVideos()
        }

        if (subscriptions.isEmpty()) {
            subscriptionsViewModel.loadSubscriptions()
        }
    }

    if (isLogged) {
        VideoList(
            ArrayList(videos),
            header = {
                SubscriptionIconList(subscriptions) { c ->
                    navController.navigate(Routes.Channel.nav(c.getFullName()))
                }
            },
            isLoading = isLoading,
            onRefresh = {
                if (!isLoading) {
                    subscriptionsViewModel.refresh()
                }
            },
            onLoadMore = {
                if (!isLoading) {
                    subscriptionsViewModel.loadVideos()
                }
            },
            onChannelClick = {
                navController.navigate(Routes.Channel.nav(it))
            }
        ) {
            navController.navigate("video/$it")
        }
    } else {
        LoginBanner()
    }
}
