package org.libre.agosto.p2play.ui.components.molecules

import android.app.DownloadManager
import android.content.Context
import android.content.Intent
import android.os.Environment
import android.widget.Toast
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.width
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.SegmentedButton
import androidx.compose.material3.SegmentedButtonDefaults
import androidx.compose.material3.SingleChoiceSegmentedButtonRow
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.R
import org.libre.agosto.p2play.domain.entities.VideoModel
import org.libre.agosto.p2play.ui.components.atoms.InputDialog
import androidx.core.net.toUri

@Composable
fun VideoActions (
    video: VideoModel,
    liked: Boolean,
    disliked: Boolean,
    onRate: (String) -> Unit,
    onReport: (String) -> Unit
) {
    val isLogged = ManagerSingleton.isLogged()
    var showReportDialog by rememberSaveable { mutableStateOf(false) }
    val context = LocalContext.current
    val shareTxt = stringResource(R.string.share_title)
    val loginFirstMsg = stringResource(R.string.pleaseLogin)

    Row(
        Modifier.fillMaxWidth(),
        verticalAlignment = Alignment.CenterVertically
    ) {
        SingleChoiceSegmentedButtonRow {
            SegmentedButton(
                liked,
                {
                    if (isLogged) {
                        onRate("like")
                    } else {
                        ManagerSingleton.toast(loginFirstMsg, context)
                    }
                },
                SegmentedButtonDefaults.itemShape(0, 2)
            ) {
                Icon(painterResource(R.drawable.ic_like), stringResource(R.string.likeBtn))
            }
            SegmentedButton(
                disliked,
                {
                    if (isLogged) {
                        onRate("dislike")
                    } else {
                        ManagerSingleton.toast(loginFirstMsg, context)
                    }
                },
                SegmentedButtonDefaults.itemShape(1, 2)
            ) {
                Icon(painterResource(R.drawable.ic_dislike), stringResource(R.string.dislikeBtn))
            }
        }

        Spacer(Modifier.width(12.dp))

        IconButton({
            val shareIntent = Intent().apply {
                action = Intent.ACTION_SEND
                putExtra(Intent.EXTRA_TEXT, "${video.name} - ${video.getVideoUrl()}")
                type = "text/plain"
            }
            context.startActivity(Intent.createChooser(shareIntent, shareTxt))
        }) {
            Icon(painterResource(R.drawable.ic_menu_share), stringResource(R.string.shareBtn))
        }

        Spacer(Modifier.width(12.dp))

        IconButton({
            // How can I download the downloadFile wutg the fileName. AI?
            val downloadFile = video.streamingData?.downloadFiles?.first()
            if (downloadFile != null) {
                val fileName = "${video.name}-${downloadFile.resolution}.${downloadFile.url.split('.').last()}"

                // Create download request
                val request = DownloadManager.Request(downloadFile.url.toUri())
                    .setTitle(video.name)
                    .setDescription("Downloading video")
                    .setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED)
                    .setDestinationInExternalPublicDir(
                        Environment.DIRECTORY_DOWNLOADS,
                        fileName
                    )
                    .setAllowedOverMetered(true)
                    .setAllowedOverRoaming(true)

                // Get DownloadManager service and enqueue download
                val downloadManager = context.getSystemService(Context.DOWNLOAD_SERVICE) as DownloadManager
                downloadManager.enqueue(request)

                // Optional: Show a toast message
                Toast.makeText(context, "Download started: $fileName", Toast.LENGTH_SHORT).show()
            } else {
                // Handle case where no download files are available
                Toast.makeText(context, "No downloadable file available", Toast.LENGTH_SHORT).show()
            }
        }) {
            Icon(
                painterResource(R.drawable.ic_outline_cloud_download_24),
                stringResource(R.string.downloadText)
            )
        }

        if (isLogged) {
            Spacer(Modifier.width(12.dp))
            IconButton({ showReportDialog = true }) {
                Icon(painterResource(R.drawable.ic_alert), stringResource(R.string.reportBtn))
            }
        }
    }

    AnimatedVisibility(showReportDialog) {
        InputDialog(stringResource(R.string.reportDialog), onDismiss = { showReportDialog = false }) {
            onReport(it)
        }
    }
}
