package org.libre.agosto.p2play.activities

import android.annotation.SuppressLint
import android.content.SharedPreferences
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.viewModels
import androidx.compose.animation.fadeOut
import androidx.compose.animation.slideInVertically
import androidx.compose.animation.slideOutVertically
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.widthIn
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import androidx.constraintlayout.compose.ConstraintLayout
import androidx.core.splashscreen.SplashScreen.Companion.installSplashScreen
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.media3.exoplayer.ExoPlayer
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import androidx.preference.PreferenceManager
import androidx.room.Room
import kotlinx.coroutines.launch
import org.libre.agosto.p2play.AppDatabase
import org.libre.agosto.p2play.Database
import org.libre.agosto.p2play.ManagerSingleton
import org.libre.agosto.p2play.activities.ui.theme.P2playTheme
import org.libre.agosto.p2play.activities.ui.theme.onErrorContainerLight
import org.libre.agosto.p2play.domain.enums.VideoFilterEnum
import org.libre.agosto.p2play.helpers.DetectActivity
import org.libre.agosto.p2play.singletons.PlaybackSingleton
import org.libre.agosto.p2play.ui.Routes
import org.libre.agosto.p2play.ui.bars.MainNavigationBar
import org.libre.agosto.p2play.ui.bars.MainTopAppBar
import org.libre.agosto.p2play.ui.bars.NotificationsBar
import org.libre.agosto.p2play.ui.bars.SearchTopBar
import org.libre.agosto.p2play.ui.components.atoms.MiniPlayer
import org.libre.agosto.p2play.ui.views.channelView.ChannelView
import org.libre.agosto.p2play.ui.views.channelView.ChannelViewModel
import org.libre.agosto.p2play.ui.views.ChooseInstanceView
import org.libre.agosto.p2play.ui.views.HomeView
import org.libre.agosto.p2play.ui.views.NotificationsView
import org.libre.agosto.p2play.ui.views.SearchView
import org.libre.agosto.p2play.ui.views.SubscriptionsView
import org.libre.agosto.p2play.ui.views.VideoListView
import org.libre.agosto.p2play.ui.views.YouView
import org.libre.agosto.p2play.ui.views.accountView.AccountView
import org.libre.agosto.p2play.ui.views.accountView.AccountViewModel
import org.libre.agosto.p2play.ui.views.videoView.VideoView
import org.libre.agosto.p2play.ui.views.videoView.VideoViewModel
import org.libre.agosto.p2play.viewModels.InstancesViewModel
import org.libre.agosto.p2play.viewModels.NotificationsViewModel
import org.libre.agosto.p2play.viewModels.SearchViewModel
import org.libre.agosto.p2play.viewModels.SessionsViewModel
import org.libre.agosto.p2play.viewModels.SubscriptionsViewModel
import org.libre.agosto.p2play.viewModels.VideosViewModel

class MainActivity : ComponentActivity() {
    private lateinit var settings: SharedPreferences
    private lateinit var db: AppDatabase
    private val videoViewModel: VideosViewModel by viewModels()
    private val subscriptionsViewModel: SubscriptionsViewModel by viewModels()
    private val searchViewModel: SearchViewModel by viewModels()
    private val notificationsViewModel: NotificationsViewModel by viewModels()
    private lateinit var instancesViewModel: InstancesViewModel
    private lateinit var sessionViewModel: SessionsViewModel

    private var isLoading = true

    @SuppressLint("RestrictedApi")
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        db = Room.databaseBuilder(
            this,
            AppDatabase::class.java,
            "p2play_v2"
        ).build()
        settings = PreferenceManager.getDefaultSharedPreferences(this)
        sessionViewModel = SessionsViewModel(db, settings, Database(this))
        instancesViewModel = InstancesViewModel(settings)

        sessionViewModel.isLoading.observe(this) {
            isLoading = it
            if (!it) {
                ManagerSingleton.settings = settings
                ManagerSingleton.loadConfigs()
            }
        }

        lifecycleScope.launch {
            sessionViewModel.load()
        }

        val splash = installSplashScreen()
        splash.setKeepOnScreenCondition({ isLoading })

        enableEdgeToEdge()

        setContent {
            val isLoadingState by sessionViewModel.isLoading.observeAsState(true)

            if (isLoadingState) {
                return@setContent
            }

            val navController = rememberNavController()
            val currentRoute = navController.currentBackStackEntryAsState().value?.destination?.route
            val routeValue = Routes.fromRoute(currentRoute)

            var player by rememberSaveable { mutableStateOf<ExoPlayer?>(null) }
            var hasBottomBar by rememberSaveable { mutableStateOf(false) }
            val notifications by notificationsViewModel.unreadNotifications.observeAsState(listOf())
            val initialRoute = if (sessionViewModel.instance.value !==
                null
            ) {
                Routes.Videos.route
            } else {
                Routes.ChooseInstance.route
            }

            var logCounter = 0

            LaunchedEffect(Unit) {
                navController.addOnDestinationChangedListener { _, _, _ ->
                    val routes = navController
                        .currentBackStack.value.joinToString(", ") { it.destination.route.toString() }

                    // TODO: Add if video is loading or buffering
                    player = if (PlaybackSingleton.isPlaying() || PlaybackSingleton.player?.isLoading == true) {
                        PlaybackSingleton.player
                    } else {
                        null
                    }

                    Log.d("BackStackLog", "[${logCounter++}] BackStack: $routes")
                }
            }

            DetectActivity(
                {
                    if (PlaybackSingleton.isPlaying()) {
                        player = PlaybackSingleton.player
                    }
                    sessionViewModel.isLogged.value?.let {
                        if (it) {
                            notificationsViewModel.refresh()
                        }
                        sessionViewModel.loadSession()
                    }
                },
                {
                    player = null
                }
            )

            P2playTheme {
                Scaffold(
                    modifier = Modifier.fillMaxSize(),
                    topBar = {
                        when (currentRoute) {
                            Routes.Search.route -> SearchTopBar(navController) {
                                searchViewModel.onSearch(it)
                            }
                            Routes.Notifications.route -> {
                                NotificationsBar(navController, {
                                    notificationsViewModel.onMarkAllAsRead()
                                })
                            }
                            Routes.Videos.route,
                                Routes.You.route,
                            Routes.Subscriptions.route -> MainTopAppBar(
                                navController,
                                Modifier,
                                sessionViewModel,
                                notifications
                            )
                            else -> {}
                        }
                    },
                    bottomBar = {
                        when (currentRoute) {
                            Routes.Videos.route,
                            Routes.You.route,
                            Routes.Subscriptions.route -> {
                                MainNavigationBar(navController)
                                hasBottomBar = true
                            }
                            else -> hasBottomBar = false
                        }
                    }
                ) { innerPadding ->
                    var topPadding = innerPadding.calculateTopPadding()

                    routeValue?.let {
                        if (it.hasTopAppBar) {
                            topPadding = 0.dp
                        }
                    }

                    ConstraintLayout(modifier = Modifier.fillMaxSize()) {
                        val miniPlayer = createRef()
                        NavHost(
                            navController,
                            startDestination = initialRoute,
                            modifier = Modifier.padding(
                                top = topPadding,
                                bottom = innerPadding.calculateBottomPadding()
                            )
                        ) {
                            composable(Routes.Videos.route) {
                                HomeView(navController, videoViewModel)
                            }
                            composable(Routes.Subscriptions.route) {
                                SubscriptionsView(navController, subscriptionsViewModel, sessionViewModel)
                            }
                            composable(Routes.Search.route) { SearchView(navController, searchViewModel) }
                            composable(Routes.You.route) {
                                YouView(navController, sessionViewModel)
                            }
                            composable(Routes.VideoList.route) { navBackStackEntry ->
                                val videoListViewModel: VideosViewModel = viewModel()
                                val key =
                                    navBackStackEntry.arguments?.getString("key")
                                        ?: VideoFilterEnum.MY_VIDEOS.toString()
                                VideoListView(videoListViewModel, key, navController)
                            }
                            composable(Routes.Notifications.route) {
                                NotificationsView(navController, notificationsViewModel)
                            }
                            composable(Routes.ChooseInstance.route) {
                                ChooseInstanceView(sessionViewModel, instancesViewModel)
                            }
                            composable(
                                Routes.VideoPlayer.route,
                                arguments = Routes.VideoPlayer.arguments,
                                enterTransition = { slideInVertically { it } },
                                exitTransition = { slideOutVertically() { it } + fadeOut() }
                            ) { backStackEntry ->
                                val videoId = backStackEntry.arguments?.getString("id") ?: ""
                                val videoViewModel: VideoViewModel = viewModel()
                                Surface(color = MaterialTheme.colorScheme.background) {
                                    VideoView(navController, videoViewModel, videoId)
                                }
                            }
                            composable(Routes.Channel.route) { backStackEntry ->
                                val channelId = backStackEntry.arguments?.getString("channelId") ?: ""
                                val channelViewModel: ChannelViewModel = viewModel()
                                ChannelView(navController, channelViewModel, channelId)
                            }
                            composable(Routes.Account.route) { backStackEntry ->
                                val accountId = backStackEntry.arguments?.getString("accountId") ?: ""
                                val accountViewModel: AccountViewModel = viewModel()
                                AccountView(navController, accountViewModel, accountId)
                            }
                        }

                        if (player !== null && hasBottomBar) {
                            Box(
                                modifier = Modifier
                                    .widthIn(max = 150.dp)
                                    .heightIn(max = 400.dp)
                                    .padding(innerPadding)
                                    .padding(bottom = 10.dp, end = 10.dp)
                                    // .background(onErrorContainerLight)
                                    .clickable {
//                                        val intent =
//                                            Intent(context, ReproductorActivity::class.java)
//                                        intent.putExtra("resume", true)
//                                        startActivity(intent)
                                        navController.navigate(
                                            Routes.VideoPlayer.nav(
                                                PlaybackSingleton.video?.uuid!!,
                                                true
                                            )
                                        )
                                    }
                                    .constrainAs(miniPlayer) {
                                        end.linkTo(parent.end)
                                        bottom.linkTo(parent.bottom)
                                    },
                                contentAlignment = Alignment.BottomEnd
                            ) {
                                MiniPlayer({ player })
                            }
                        }
                    }
                }
            }
        }
    }
}
