/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { nearestDivisor } from '../util';
import { PARALLELIZE_THRESHOLD } from './reduce_util';
export function segOpComputeOptimalWindowSize(inSize, numSegments) {
    let done = false;
    let res;
    if (inSize <= PARALLELIZE_THRESHOLD) {
        res = inSize;
        done = true;
    }
    else {
        res = nearestDivisor(inSize, Math.floor(Math.sqrt(inSize)));
    }
    while (!done) {
        if (res > numSegments || res === inSize) {
            done = true;
        }
        else {
            res = nearestDivisor(inSize, res + 1);
        }
    }
    return res;
}
export function computeOutShape(aShape, axis, numSegments) {
    const outShape = [];
    const rank = aShape.length;
    for (let dim = 0; dim < rank; dim++) {
        if (dim !== axis) {
            outShape.push(aShape[dim]);
        }
        else {
            outShape.push(numSegments);
        }
    }
    return outShape;
}
export function collectGatherOpShapeInfo(x, indices, axis, batchDims) {
    const indicesRank = indices.shape.length;
    const xRank = x.shape.length;
    if (batchDims !== 0) {
        if (batchDims < -indicesRank || batchDims > indicesRank) {
            throw new Error(`Expect batchDims in the range of [-${indicesRank}, ${indicesRank}], but got ${batchDims}`);
        }
    }
    if (batchDims < 0) {
        batchDims += indicesRank;
    }
    if (batchDims > xRank) {
        throw new Error(`batchDims (${batchDims}) must be less than rank(x) (
    ${xRank}).`);
    }
    if (axis < batchDims) {
        throw new Error(`batchDims (${batchDims}) must be less than or equal to axis (${axis}).`);
    }
    for (let i = 0; i < batchDims; ++i) {
        if (x.shape[i] !== indices.shape[i]) {
            throw new Error(`x.shape[${i}]: ${x.shape[i]} should be equal to indices.shape[${i}]: ${indices.shape[i]}.`);
        }
    }
    const dimSize = x.shape[axis];
    const outputShape = [];
    let batchSize = 1;
    let outerSize = 1;
    let sliceSize = 1;
    for (let i = 0; i < batchDims; ++i) {
        outputShape.push(x.shape[i]);
        batchSize *= x.shape[i];
    }
    for (let i = batchDims; i < axis; i++) {
        outputShape.push(x.shape[i]);
        outerSize *= x.shape[i];
    }
    for (let i = batchDims; i < indicesRank; i++) {
        outputShape.push(indices.shape[i]);
    }
    for (let i = axis + 1; i < xRank; i++) {
        outputShape.push(x.shape[i]);
        sliceSize *= x.shape[i];
    }
    return { batchSize, sliceSize, outerSize, dimSize, outputShape };
}
//# sourceMappingURL=data:application/json;base64,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