/**
 * @license
 * Copyright 2020 Google Inc. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { BROWSER_ENVS, describeWithFlags, NODE_ENVS } from '../jasmine_util';
import { expectArraysClose, expectArraysEqual } from '../test_util';
class MockContext {
    getImageData(x, y, width, height) {
        const data = new Uint8ClampedArray(width * height * 4);
        for (let i = 0; i < data.length; ++i) {
            data[i] = i + 1;
        }
        return { data };
    }
}
class MockCanvas {
    constructor(width, height) {
        this.width = width;
        this.height = height;
    }
    getContext(type) {
        return new MockContext();
    }
}
describeWithFlags('fromPixels, mock canvas', NODE_ENVS, () => {
    it('accepts a canvas-like element', async () => {
        const c = new MockCanvas(2, 2);
        // tslint:disable-next-line:no-any
        const t = tf.browser.fromPixels(c);
        expect(t.dtype).toBe('int32');
        expect(t.shape).toEqual([2, 2, 3]);
        expectArraysEqual(await t.data(), [1, 2, 3, 5, 6, 7, 9, 10, 11, 13, 14, 15]);
    });
    it('accepts a canvas-like element, numChannels=4', async () => {
        const c = new MockCanvas(2, 2);
        // tslint:disable-next-line:no-any
        const t = tf.browser.fromPixels(c, 4);
        expect(t.dtype).toBe('int32');
        expect(t.shape).toEqual([2, 2, 4]);
        expectArraysEqual(await t.data(), [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16]);
    });
    it('errors when passed a non-canvas object', () => {
        // tslint:disable-next-line:no-any
        expect(() => tf.browser.fromPixels(5)).toThrowError();
    });
});
describeWithFlags('fromPixels', BROWSER_ENVS, () => {
    it('ImageData 1x1x3', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        const array = tf.browser.fromPixels(pixels, 3);
        expectArraysEqual(await array.data(), [0, 80, 160]);
    });
    it('ImageData 1x1x4', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        const array = tf.browser.fromPixels(pixels, 4);
        expectArraysEqual(await array.data(), [0, 80, 160, 240]);
    });
    it('ImageData 2x2x3', async () => {
        const pixels = new ImageData(2, 2);
        for (let i = 0; i < 8; i++) {
            pixels.data[i] = i * 2;
        }
        for (let i = 8; i < 16; i++) {
            pixels.data[i] = i * 2;
        }
        const array = tf.browser.fromPixels(pixels, 3);
        expectArraysEqual(await array.data(), [0, 2, 4, 8, 10, 12, 16, 18, 20, 24, 26, 28]);
    });
    it('ImageData 2x2x4', async () => {
        const pixels = new ImageData(2, 2);
        for (let i = 0; i < 8; i++) {
            pixels.data[i] = i * 2;
        }
        for (let i = 8; i < 16; i++) {
            pixels.data[i] = i * 2;
        }
        const array = tf.browser.fromPixels(pixels, 4);
        expectArraysClose(await array.data(), new Int32Array([0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30]));
    });
    it('fromPixels, 3 channels', async () => {
        const pixels = new ImageData(1, 2);
        pixels.data[0] = 2;
        pixels.data[1] = 3;
        pixels.data[2] = 4;
        pixels.data[3] = 255; // Not used.
        pixels.data[4] = 5;
        pixels.data[5] = 6;
        pixels.data[6] = 7;
        pixels.data[7] = 255; // Not used.
        const res = tf.browser.fromPixels(pixels, 3);
        expect(res.shape).toEqual([2, 1, 3]);
        expect(res.dtype).toBe('int32');
        expectArraysClose(await res.data(), [2, 3, 4, 5, 6, 7]);
    });
    it('fromPixels, reshape, then do tf.add()', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 2;
        pixels.data[1] = 3;
        pixels.data[2] = 4;
        pixels.data[3] = 255; // Not used.
        const a = tf.browser.fromPixels(pixels, 3).reshape([1, 1, 1, 3]);
        const res = a.add(tf.scalar(2, 'int32'));
        expect(res.shape).toEqual([1, 1, 1, 3]);
        expect(res.dtype).toBe('int32');
        expectArraysClose(await res.data(), [4, 5, 6]);
    });
    it('fromPixels + fromPixels', async () => {
        const pixelsA = new ImageData(1, 1);
        pixelsA.data[0] = 255;
        pixelsA.data[1] = 3;
        pixelsA.data[2] = 4;
        pixelsA.data[3] = 255; // Not used.
        const pixelsB = new ImageData(1, 1);
        pixelsB.data[0] = 5;
        pixelsB.data[1] = 6;
        pixelsB.data[2] = 7;
        pixelsB.data[3] = 255; // Not used.
        const a = tf.browser.fromPixels(pixelsA, 3).toFloat();
        const b = tf.browser.fromPixels(pixelsB, 3).toFloat();
        const res = a.add(b);
        expect(res.shape).toEqual([1, 1, 3]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), [260, 9, 11]);
    });
    it('fromPixels for PixelData type', async () => {
        const dataA = new Uint8Array([255, 3, 4, 255]);
        const pixelsA = { width: 1, height: 1, data: dataA };
        const dataB = new Uint8Array([5, 6, 7, 255]);
        const pixelsB = { width: 1, height: 1, data: dataB };
        const a = tf.browser.fromPixels(pixelsA, 3).toFloat();
        const b = tf.browser.fromPixels(pixelsB, 3).toFloat();
        const res = a.add(b);
        expect(res.shape).toEqual([1, 1, 3]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), [260, 9, 11]);
    });
    it('fromPixels for HTMLCanvasElement', async () => {
        const canvas = document.createElement('canvas');
        canvas.width = 1;
        canvas.height = 1;
        const ctx = canvas.getContext('2d');
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        ctx.putImageData(pixels, 1, 1);
        const res = tf.browser.fromPixels(canvas);
        expect(res.shape).toEqual([1, 1, 3]);
        const data = await res.data();
        expect(data.length).toEqual(1 * 1 * 3);
    });
    it('fromPixels for HTMLImageElement', async () => {
        const img = new Image(10, 10);
        img.src = 'data:image/gif;base64' +
            ',R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        const res = tf.browser.fromPixels(img);
        expect(res.shape).toEqual([10, 10, 3]);
        const data = await res.data();
        expect(data.length).toEqual(10 * 10 * 3);
    });
    it('fromPixels for HTMLVideoElement', async () => {
        const video = document.createElement('video');
        video.autoplay = true;
        const source = document.createElement('source');
        source.src =
            // tslint:disable-next-line:max-line-length
            'data:video/mp4;base64,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';
        source.type = 'video/mp4';
        video.appendChild(source);
        document.body.appendChild(video);
        // On mobile safari the ready state is ready immediately.
        if (video.readyState < 2) {
            await new Promise(resolve => {
                video.addEventListener('loadeddata', () => resolve(video));
            });
        }
        const res = tf.browser.fromPixels(video);
        expect(res.shape).toEqual([90, 160, 3]);
        const data = await res.data();
        expect(data.length).toEqual(90 * 160 * 3);
        document.body.removeChild(video);
    });
    it('fromPixels for HTMLVideoElement throws without loadeddata', async () => {
        const video = document.createElement('video');
        video.width = 1;
        video.height = 1;
        video.src = 'data:image/gif;base64' +
            ',R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        expect(() => tf.browser.fromPixels(video)).toThrowError();
    });
    it('throws when passed a primitive number', () => {
        const msg = /pixels passed to tf.browser.fromPixels\(\) must be either/;
        // tslint:disable-next-line:no-any
        expect(() => tf.browser.fromPixels(3)).toThrowError(msg);
    });
    it('throws when passed a string', () => {
        const msg = /pixels passed to tf.browser.fromPixels\(\) must be either/;
        // tslint:disable-next-line:no-any
        expect(() => tf.browser.fromPixels('test')).toThrowError(msg);
    });
    it('canvas and image match', async () => {
        const img = new Image();
        const size = 80;
        img.src =
            // tslint:disable-next-line:max-line-length
            'data:image/jpeg;base64,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';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        img.width = size;
        img.height = size;
        const pixels = await tf.browser.fromPixels(img, 4);
        const canvas = document.createElement('canvas');
        canvas.width = size;
        canvas.height = size;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(img, 0, 0, size, size);
        const actual = ctx.getImageData(0, 0, size, size).data;
        const actualInt32 = Int32Array.from(actual);
        const pixelsData = await pixels.data();
        expectArraysClose(pixelsData, actualInt32, 10);
    });
    if (tf.env().getBool('IS_CHROME')) {
        it('fromPixels for ImageBitmap', async () => {
            const imageDataWidth = 1;
            const imageDataHeight = 2;
            const numChannel = 3;
            const pixels = new ImageData(imageDataWidth, imageDataHeight);
            for (let i = 0; i < imageDataWidth * imageDataHeight * 4; ++i) {
                if (i % 4 === 3) {
                    pixels.data[i] = 255;
                }
                else {
                    pixels.data[i] = i;
                }
            }
            const imageBitmap = await createImageBitmap(pixels);
            const res = tf.browser.fromPixels(imageBitmap, numChannel);
            imageBitmap.close();
            expect(res.shape).toEqual([imageDataHeight, imageDataWidth, numChannel]);
            const data = await res.data();
            expect(data.length)
                .toEqual(imageDataHeight * imageDataWidth * numChannel);
            expectArraysEqual(await res.data(), [0, 1, 2, 4, 5, 6]);
        });
        it('fromPixels for ImageBitmap outShape changes', async () => {
            const imageDataWidth = 2;
            const imageDataHeight = 2;
            let numChannel = 3;
            const pixels = new ImageData(imageDataWidth, imageDataHeight);
            for (let i = 0; i < imageDataWidth * imageDataHeight * 4; ++i) {
                if (i % 4 === 3) {
                    pixels.data[i] = 255;
                }
                else {
                    pixels.data[i] = i;
                }
            }
            const imageBitmap = await createImageBitmap(pixels);
            const res = tf.browser.fromPixels(imageBitmap, numChannel);
            expect(res.shape).toEqual([imageDataHeight, imageDataWidth, numChannel]);
            const data = await res.data();
            expect(data.length)
                .toEqual(imageDataHeight * imageDataWidth * numChannel);
            expectArraysEqual(await res.data(), [0, 1, 2, 4, 5, 6, 8, 9, 10, 12, 13, 14]);
            // Change output shapes
            numChannel = 4;
            const resShapeChange = tf.browser.fromPixels(imageBitmap, numChannel);
            expect(resShapeChange.shape).toEqual([
                imageDataHeight, imageDataWidth, numChannel
            ]);
            const data2 = await resShapeChange.data();
            expect(data2.length)
                .toEqual(imageDataHeight * imageDataWidth * numChannel);
            expectArraysEqual(await resShapeChange.data(), [0, 1, 2, 255, 4, 5, 6, 255, 8, 9, 10, 255, 12, 13, 14, 255]);
        });
    }
});
//# sourceMappingURL=data:application/json;base64,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