/**
 * @license
 * Copyright 2020 Google Inc. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { BROWSER_ENVS, describeWithFlags, NODE_ENVS } from '../jasmine_util';
import { expectArraysClose, expectArraysEqual } from '../test_util';
class MockContext {
    getImageData(x, y, width, height) {
        const data = new Uint8ClampedArray(width * height * 4);
        for (let i = 0; i < data.length; ++i) {
            data[i] = i + 1;
        }
        return { data };
    }
}
class MockCanvas {
    constructor(width, height) {
        this.width = width;
        this.height = height;
    }
    getContext(type) {
        return new MockContext();
    }
}
describeWithFlags('fromPixelsAsync, mock canvas', NODE_ENVS, () => {
    it('accepts a canvas-like element', async () => {
        const c = new MockCanvas(2, 2);
        // tslint:disable-next-line:no-any
        const t = await tf.browser.fromPixelsAsync(c);
        expect(t.dtype).toBe('int32');
        expect(t.shape).toEqual([2, 2, 3]);
        expectArraysEqual(await t.data(), [1, 2, 3, 5, 6, 7, 9, 10, 11, 13, 14, 15]);
    });
    it('accepts a canvas-like element, numChannels=4', async () => {
        const c = new MockCanvas(2, 2);
        // tslint:disable-next-line:no-any
        const t = await tf.browser.fromPixelsAsync(c, 4);
        expect(t.dtype).toBe('int32');
        expect(t.shape).toEqual([2, 2, 4]);
        expectArraysEqual(await t.data(), [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16]);
    });
});
// Flag 'WRAP_TO_IMAGEBITMAP' is set by customer. The default
// value is false. The cases below won't try to wrap input to
// imageBitmap.
describeWithFlags('fromPixelsAsync, ' +
    '|WRAP_TO_IMAGEBITMAP| false', BROWSER_ENVS, () => {
    it('ImageData 1x1x3', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        const array = await tf.browser.fromPixelsAsync(pixels, 3);
        expectArraysEqual(await array.data(), [0, 80, 160]);
    });
    it('ImageData 1x1x4', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        const array = await tf.browser.fromPixelsAsync(pixels, 4);
        expectArraysEqual(await array.data(), [0, 80, 160, 240]);
    });
    it('ImageData 2x2x3', async () => {
        const pixels = new ImageData(2, 2);
        for (let i = 0; i < 8; i++) {
            pixels.data[i] = i * 2;
        }
        for (let i = 8; i < 16; i++) {
            pixels.data[i] = i * 2;
        }
        const array = await tf.browser.fromPixelsAsync(pixels, 3);
        expectArraysEqual(await array.data(), [0, 2, 4, 8, 10, 12, 16, 18, 20, 24, 26, 28]);
    });
    it('ImageData 2x2x4', async () => {
        const pixels = new ImageData(2, 2);
        for (let i = 0; i < 8; i++) {
            pixels.data[i] = i * 2;
        }
        for (let i = 8; i < 16; i++) {
            pixels.data[i] = i * 2;
        }
        const array = await tf.browser.fromPixelsAsync(pixels, 4);
        expectArraysClose(await array.data(), new Int32Array([0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30]));
    });
    it('fromPixelsAsync, 3 channels', async () => {
        const pixels = new ImageData(1, 2);
        pixels.data[0] = 2;
        pixels.data[1] = 3;
        pixels.data[2] = 4;
        pixels.data[3] = 255; // Not used.
        pixels.data[4] = 5;
        pixels.data[5] = 6;
        pixels.data[6] = 7;
        pixels.data[7] = 255; // Not used.
        const res = await tf.browser.fromPixelsAsync(pixels, 3);
        expect(res.shape).toEqual([2, 1, 3]);
        expect(res.dtype).toBe('int32');
        expectArraysClose(await res.data(), [2, 3, 4, 5, 6, 7]);
    });
    it('fromPixelsAsync, reshape, then do tf.add()', async () => {
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 2;
        pixels.data[1] = 3;
        pixels.data[2] = 4;
        pixels.data[3] = 255; // Not used.
        const content = await tf.browser.fromPixelsAsync(pixels, 3);
        const a = content.reshape([1, 1, 1, 3]);
        const res = a.add(tf.scalar(2, 'int32'));
        expect(res.shape).toEqual([1, 1, 1, 3]);
        expect(res.dtype).toBe('int32');
        expectArraysClose(await res.data(), [4, 5, 6]);
    });
    it('fromPixelsAsync + fromPixelsAsync', async () => {
        const pixelsA = new ImageData(1, 1);
        pixelsA.data[0] = 255;
        pixelsA.data[1] = 3;
        pixelsA.data[2] = 4;
        pixelsA.data[3] = 255; // Not used.
        const pixelsB = new ImageData(1, 1);
        pixelsB.data[0] = 5;
        pixelsB.data[1] = 6;
        pixelsB.data[2] = 7;
        pixelsB.data[3] = 255; // Not used.
        const contentA = await tf.browser.fromPixelsAsync(pixelsA, 3);
        const contentB = await tf.browser.fromPixelsAsync(pixelsB, 3);
        const a = contentA.toFloat();
        const b = contentB.toFloat();
        const res = a.add(b);
        expect(res.shape).toEqual([1, 1, 3]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), [260, 9, 11]);
    });
    it('fromPixelsAsync for PixelData type', async () => {
        const dataA = new Uint8Array([255, 3, 4, 255]);
        const pixelsA = { width: 1, height: 1, data: dataA };
        const dataB = new Uint8Array([5, 6, 7, 255]);
        const pixelsB = { width: 1, height: 1, data: dataB };
        const contentA = await tf.browser.fromPixelsAsync(pixelsA, 3);
        const contentB = await tf.browser.fromPixelsAsync(pixelsB, 3);
        const a = contentA.toFloat();
        const b = contentB.toFloat();
        const res = a.add(b);
        expect(res.shape).toEqual([1, 1, 3]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), [260, 9, 11]);
    });
    it('fromPixelsAsync for HTMLCanvasElement', async () => {
        const canvas = document.createElement('canvas');
        canvas.width = 1;
        canvas.height = 1;
        const ctx = canvas.getContext('2d');
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        ctx.putImageData(pixels, 1, 1);
        const res = await tf.browser.fromPixelsAsync(canvas);
        expect(res.shape).toEqual([1, 1, 3]);
        const data = await res.data();
        expect(data.length).toEqual(1 * 1 * 3);
    });
    it('fromPixelsAsync for HTMLImageElement', async () => {
        const img = new Image(10, 10);
        img.src = 'data:image/gif;base64' +
            ',R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        const res = await tf.browser.fromPixelsAsync(img);
        expect(res.shape).toEqual([10, 10, 3]);
        const data = await res.data();
        expect(data.length).toEqual(10 * 10 * 3);
    });
    it('fromPixelsAsync for HTMLVideoElement', async () => {
        const video = document.createElement('video');
        video.autoplay = true;
        const source = document.createElement('source');
        source.src =
            // tslint:disable-next-line:max-line-length
            'data:video/mp4;base64,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';
        source.type = 'video/mp4';
        video.appendChild(source);
        document.body.appendChild(video);
        // On mobile safari the ready state is ready immediately.
        if (video.readyState < 2) {
            await new Promise(resolve => {
                video.addEventListener('loadeddata', () => resolve(video));
            });
        }
        const res = await tf.browser.fromPixelsAsync(video);
        expect(res.shape).toEqual([90, 160, 3]);
        const data = await res.data();
        expect(data.length).toEqual(90 * 160 * 3);
        document.body.removeChild(video);
    });
    it('canvas and image match', async () => {
        const img = new Image();
        const size = 80;
        img.src =
            // tslint:disable-next-line:max-line-length
            'data:image/jpeg;base64,/9j/4AAQSkZJRgABAQAASABIAAD/4QCMRXhpZgAATU0AKgAAAAgABQESAAMAAAABAAEAAAEaAAUAAAABAAAASgEbAAUAAAABAAAAUgEoAAMAAAABAAIAAIdpAAQAAAABAAAAWgAAAAAAAABIAAAAAQAAAEgAAAABAAOgAQADAAAAAQABAACgAgAEAAAAAQAAAFCgAwAEAAAAAQAAADwAAAAA/+0AOFBob3Rvc2hvcCAzLjAAOEJJTQQEAAAAAAAAOEJJTQQlAAAAAAAQ1B2M2Y8AsgTpgAmY7PhCfv/AABEIADwAUAMBIgACEQEDEQH/xAAfAAABBQEBAQEBAQAAAAAAAAAAAQIDBAUGBwgJCgv/xAC1EAACAQMDAgQDBQUEBAAAAX0BAgMABBEFEiExQQYTUWEHInEUMoGRoQgjQrHBFVLR8CQzYnKCCQoWFxgZGiUmJygpKjQ1Njc4OTpDREVGR0hJSlNUVVZXWFlaY2RlZmdoaWpzdHV2d3h5eoOEhYaHiImKkpOUlZaXmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4eLj5OXm5+jp6vHy8/T19vf4+fr/xAAfAQADAQEBAQEBAQEBAAAAAAAAAQIDBAUGBwgJCgv/xAC1EQACAQIEBAMEBwUEBAABAncAAQIDEQQFITEGEkFRB2FxEyIygQgUQpGhscEJIzNS8BVictEKFiQ04SXxFxgZGiYnKCkqNTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqCg4SFhoeIiYqSk5SVlpeYmZqio6Slpqeoqaqys7S1tre4ubrCw8TFxsfIycrS09TV1tfY2dri4+Tl5ufo6ery8/T19vf4+fr/2wBDAAkGBxMSEhUSEhIVFRUXFxUWFRUVFRUVDxUVFhUWFxUVFRUYHSggGBolGxUVITEhJSkrLi4uFx8zODMtNygtLiv/2wBDAQoKCg4NDhsQEBotIB8fLS0tLS0tLS0tLSstLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLSstLS0tLS3/3QAEAAX/2gAMAwEAAhEDEQA/APP/AAlPI3nFOX2g5J9O5roPDuouWZJpEPdSCM1ydxbeXCWUtuzjKE42nrnFNtrlR5eACV5wRyOPWtYyWg1C7sehavfNEu8OFGO4zn6Vk6JczyOpWQu0p4P8KDvkdgACawdfcvGuX98A5rp/CMe22mQpt2x9f4mLhi2fToKKk+VN/cV7K0kt7nS6cXJXcjlWLASFlCnHQ4HI3dvwputWG7Dxu0bKRkg/Kc9AynsemeoNOOtrJE4gUyFBjA4BI4wD7GqxvG2q0qFGIKsD3Ddf1ANccK8m7s2qUEl7pUa8lZ9iuy9skAjI681vW68DPXFcxfXKxMkhJ5by/wDZzWsl43mBcjHpjnGOtd0Jc2pySVmbPlinooxVdZKej1oyD//Q8lstTkh3AdCCpBGR6VDHcYx6jv7V21zYxQwkjBcck9VOeoKmsSNY5QRsAUAkYGMYq3oPU2Bpm5IZThdwXI4HPUGtjUw8Fo5b77A4AHXsC3sM1zXhmBJnKzMxQLwuT1zXZarajyAuSQ2doPJCAd/bjH1NZ1pLk+42hzSkmyXQ9Y86FTCqoCqhiAvDfxbvQ5HoaNZL7Pnb7xwg5znHB55Jzz0rlvBUMgusxllTygXx93dwF9ieDWlfW8hulMkpf72zcMbSQRxjjvXDzJStf0OxXlG9hdQTzrafA5GHUf7SAMB/MfjWFB4pdYEDDMgyUkIHKZ4B/Sup05MCRO6OQR/skDH4EVkWVgjyfZTHlG3FW/uLnkZ+prtoVZJNI4akFc6LQ7rzVWVWDJjB9Q/cGrkuqRxsqM2Gbp/+usW60g2kJSNmaLfuYA8j8fSqEOsrzG4yB8xxgkDqOa6ee7sYch//0fMtOuDJIInYlMngntnpmtLxLAIpEQfLCyjheOh5GfyrNvLD7PdiJHDdCGIx1zwfyrS8SxGWSBQ64bCbifkVu+TWnLvcaegonjtfLaL5i567uQnAx+ddolpJekpG2yMffkI56YCqvtzjt39jxv8AYASdbeSXzM42tAAwG4ng5zt6dTXrGl24iiwP/r+nPvWGJ3S7G+Hd7lOLTUhUJENpAAB67iOhcd6rXEIlGdoWRTyOpVhzwe4PY1ZeYCQZPU4FVdfnMTxzJ3yjDs4ALAH8jz2zXPJRO2jGU3yLfp/kZ1zIuR1SQ8EjGTjsQeoqtYp5dxznJUkE8AqTzWvqCLPEJIjhgcg/xKw6hhWUsrltsmAwHy5IP3vQnnFXR9yVns+pzVqb16NdB+oXjMjgcjDcV5Q90d5ZcjPHXnHpXsslioh46kfqRXi9yhV2B6hmB+oJBrskrHHe5//S8la4Z5leYdSuR0yAea69NLQzKjRZgJ3oCc4IHII9DmsCOzWVyGzwuRg4rtbVf9WPRTz36CuujCLun0sQ20tDkTKbeVntVCkb0KkE7iTkAAfQY+tevwlhCm772xd31wM/rXiuoyst4wV2GJRjHYkqCf1Ne43R4rhxSVzswz3OWvyTcQrkj5iT7jGP61F4o1JHKRJyI8lj23Ebdo+gzn3xWP4vnYXcYBI+U9OD1HeqJriq6SPby+kv4j6Ghb6g8R3I2OxB5Vh6MO9PmvzNJGGUDa3AGe/qe49qyC1afh+MNcID2BP4ggf1NaUr3SNsWoNSm46pM3bm8wMd815RqaFppmUEgOxPtz/jXsuuWCIRtzyCfYfT2ryTxMNlxIq8BtpIHQk5r0JM+VtY/9k=';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        img.width = size;
        img.height = size;
        const pixels = await tf.browser.fromPixelsAsync(img, 4);
        const canvas = document.createElement('canvas');
        canvas.width = size;
        canvas.height = size;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(img, 0, 0, size, size);
        const actual = ctx.getImageData(0, 0, size, size).data;
        const actualInt32 = Int32Array.from(actual);
        const pixelsData = await pixels.data();
        expectArraysClose(pixelsData, actualInt32, 10);
    });
});
// Flag |WRAP_TO_IMAGEBITMAP| is set by customer.
// This flag helps on image, video and canvas input cases
// for WebGPU backends. We'll cover these inputs with test
// cases set 'WRAP_TO_IMAGEBITMAP' to true.
describeWithFlags('fromPixelsAsync, ' +
    '|WRAP_TO_IMAGEBITMAP| true', BROWSER_ENVS, () => {
    beforeAll(() => {
        tf.env().set('WRAP_TO_IMAGEBITMAP', true);
    });
    afterAll(() => {
        tf.env().set('WRAP_TO_IMAGEBITMAP', false);
    });
    it('fromPixelsAsync for HTMLCanvasElement ', async () => {
        const canvas = document.createElement('canvas');
        canvas.width = 1;
        canvas.height = 1;
        const ctx = canvas.getContext('2d');
        const pixels = new ImageData(1, 1);
        pixels.data[0] = 0;
        pixels.data[1] = 80;
        pixels.data[2] = 160;
        pixels.data[3] = 240;
        ctx.putImageData(pixels, 1, 1);
        const res = await tf.browser.fromPixelsAsync(canvas);
        expect(res.shape).toEqual([1, 1, 3]);
        const data = await res.data();
        expect(data.length).toEqual(1 * 1 * 3);
    });
    it('fromPixelsAsync for HTMLImageElement', async () => {
        const img = new Image(10, 10);
        img.src = 'data:image/gif;base64' +
            ',R0lGODlhAQABAIAAAP///wAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        const res = await tf.browser.fromPixelsAsync(img);
        expect(res.shape).toEqual([10, 10, 3]);
        const data = await res.data();
        expect(data.length).toEqual(10 * 10 * 3);
    });
    it('fromPixelsAsync for HTMLVideoElement', async () => {
        const video = document.createElement('video');
        video.autoplay = true;
        const source = document.createElement('source');
        source.src =
            // tslint:disable-next-line:max-line-length
            'data:video/mp4;base64,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';
        source.type = 'video/mp4';
        video.appendChild(source);
        document.body.appendChild(video);
        // On mobile safari the ready state is ready immediately.
        if (video.readyState < 2) {
            await new Promise(resolve => {
                video.addEventListener('loadeddata', () => resolve(video));
            });
        }
        const res = await tf.browser.fromPixelsAsync(video);
        expect(res.shape).toEqual([90, 160, 3]);
        const data = await res.data();
        expect(data.length).toEqual(90 * 160 * 3);
        document.body.removeChild(video);
    });
    it('canvas and image match', async () => {
        const img = new Image();
        const size = 80;
        img.src =
            // tslint:disable-next-line:max-line-length
            'data:image/jpeg;base64,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';
        await new Promise(resolve => {
            img.onload = () => resolve(img);
        });
        img.width = size;
        img.height = size;
        const pixels = await tf.browser.fromPixelsAsync(img, 4);
        const canvas = document.createElement('canvas');
        canvas.width = size;
        canvas.height = size;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(img, 0, 0, size, size);
        const actual = ctx.getImageData(0, 0, size, size).data;
        const actualInt32 = Int32Array.from(actual);
        const pixelsData = await pixels.data();
        expectArraysClose(pixelsData, actualInt32, 10);
    });
});
//# sourceMappingURL=data:application/json;base64,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