/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, BROWSER_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
describeWithFlags('depthToSpace', ALL_ENVS, () => {
    it('tensor4d, input shape=[1, 1, 1, 4], blockSize=2, format=NHWC', async () => {
        const t = tf.tensor4d([[[[1, 2, 3, 4]]]]);
        const blockSize = 2;
        const dataFormat = 'NHWC';
        const res = tf.depthToSpace(t, blockSize, dataFormat);
        expect(res.shape).toEqual([1, 2, 2, 1]);
        expectArraysClose(await res.data(), [1, 2, 3, 4]);
    });
    it('tensor4d, input shape=[1, 1, 1, 12], blockSize=2, format=NHWC', async () => {
        const t = tf.tensor4d([[[[1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]]]]);
        const blockSize = 2;
        const dataFormat = 'NHWC';
        const res = tf.depthToSpace(t, blockSize, dataFormat);
        expect(res.shape).toEqual([1, 2, 2, 3]);
        expectArraysClose(await res.data(), [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]);
    });
    it('tensor4d, input shape=[1, 2, 2, 4], blockSize=2, format=NHWC', async () => {
        const t = tf.tensor4d([
            [[[1, 2, 3, 4], [5, 6, 7, 8]], [[9, 10, 11, 12], [13, 14, 15, 16]]]
        ]);
        const blockSize = 2;
        const dataFormat = 'NHWC';
        const res = tf.depthToSpace(t, blockSize, dataFormat);
        expect(res.shape).toEqual([1, 4, 4, 1]);
        expectArraysClose(await res.data(), [1, 2, 5, 6, 3, 4, 7, 8, 9, 10, 13, 14, 11, 12, 15, 16]);
    });
    it('throws when depth not divisible by blockSize * blockSize', () => {
        const t = tf.tensor4d([1, 2, 3, 4], [1, 1, 1, 4]);
        const blockSize = 3;
        expect(() => tf.depthToSpace(t, blockSize))
            .toThrowError(`Dimension size must be evenly divisible by ${blockSize * blockSize} but is ${t.shape[3]} for depthToSpace with input shape ${t.shape}`);
    });
    it('throws for int32 input', async () => {
        const t = tf.tensor4d([[[[1, 2, 3, 4]]]], [1, 1, 1, 4], 'int32');
        const blockSize = 2;
        const dataFormat = 'NHWC';
        expect(() => tf.depthToSpace(t, blockSize, dataFormat))
            .toThrowError(/Argument 'x' passed to 'depthToSpace' must be float32/);
    });
});
describeWithFlags('depthToSpace', BROWSER_ENVS, () => {
    it('throws when blocksize < 2', () => {
        const t = tf.tensor4d([1, 2, 3, 4], [1, 1, 1, 4]);
        const blockSize = 1;
        expect(() => tf.depthToSpace(t, blockSize))
            .toThrowError(`blockSize should be > 1 for depthToSpace, but was: ${blockSize}`);
    });
});
//# sourceMappingURL=data:application/json;base64,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