/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
import { sizeFromShape } from '../util';
// Generates small floating point inputs to avoid overflows
function generateCaseInputs(totalSizeTensor, totalSizeFilter) {
    const inp = new Array(totalSizeTensor);
    const filt = new Array(totalSizeFilter);
    for (let i = 0; i < totalSizeTensor; i++) {
        inp[i] = (i + 1) / totalSizeTensor;
    }
    for (let i = 0; i < totalSizeFilter; i++) {
        filt[i] = (i + 1) / totalSizeFilter;
    }
    return { input: inp, filter: filt };
}
function generateGradientCaseInputs(totalSizeTensor, totalSizeFilter) {
    const inp = new Array(totalSizeTensor);
    const filt = new Array(totalSizeFilter);
    for (let i = 0; i < totalSizeTensor; i++) {
        inp[i] = i + 1;
    }
    for (let i = 0; i < totalSizeFilter; i++) {
        filt[i] = i + 1;
    }
    return { input: inp, filter: filt };
}
function runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fDepth, fHeight, fWidth, pad, stride) {
    const inputShape = [batch, inDepth, inHeight, inWidth, inChannels];
    const filterShape = [fDepth, fHeight, fWidth, inChannels, outChannels];
    const totalSizeTensor = sizeFromShape(inputShape);
    const totalSizeFilter = sizeFromShape(filterShape);
    const inputs = generateCaseInputs(totalSizeTensor, totalSizeFilter);
    const x = tf.tensor5d(inputs.input, inputShape);
    const w = tf.tensor5d(inputs.filter, filterShape);
    const result = tf.conv3d(x, w, stride, pad);
    return result;
}
function runGradientConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fDepth, fHeight, fWidth, pad, stride) {
    const inputShape = [batch, inDepth, inHeight, inWidth, inChannels];
    const filterShape = [fDepth, fHeight, fWidth, inChannels, outChannels];
    const totalSizeTensor = sizeFromShape(inputShape);
    const totalSizeFilter = sizeFromShape(filterShape);
    const inputs = generateGradientCaseInputs(totalSizeTensor, totalSizeFilter);
    const x = tf.tensor5d(inputs.input, inputShape);
    const w = tf.tensor5d(inputs.filter, filterShape);
    const grads = tf.grads((x, filter) => tf.conv3d(x.clone(), filter.clone(), stride, pad).clone());
    const [dx, dfilter] = grads([x, w]);
    expect(dx.shape).toEqual(x.shape);
    expect(dfilter.shape).toEqual(w.shape);
    return [dx, dfilter];
}
describeWithFlags('conv3d', ALL_ENVS, () => {
    it('x=[1, 2, 3, 1, 3] f=[1, 1, 1, 3, 3] s=1 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 2;
        const inHeight = 3;
        const inWidth = 1;
        const inChannels = 3;
        const outChannels = 3;
        const fSize = 1;
        const pad = 'valid';
        const stride = 1;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            0.18518519, 0.22222222, 0.25925926, 0.40740741, 0.5, 0.59259259,
            0.62962963, 0.77777778, 0.92592593, 0.85185185, 1.05555556, 1.25925926,
            1.07407407, 1.33333333, 1.59259259, 1.2962963, 1.61111111, 1.92592593
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 2, 1, 3, 3] f=[1, 1, 1, 3, 3] s=1 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 2;
        const inHeight = 1;
        const inWidth = 3;
        const inChannels = 3;
        const outChannels = 3;
        const fSize = 1;
        const pad = 'valid';
        const stride = 1;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            0.18518519, 0.22222222, 0.25925926, 0.40740741, 0.5, 0.59259259,
            0.62962963, 0.77777778, 0.92592593, 0.85185185, 1.05555556, 1.25925926,
            1.07407407, 1.33333333, 1.59259259, 1.2962963, 1.61111111, 1.92592593
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 1, 2, 3, 3] f=[1, 1, 1, 3, 3] s=1 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 1;
        const inHeight = 2;
        const inWidth = 3;
        const inChannels = 3;
        const outChannels = 3;
        const fSize = 1;
        const pad = 'valid';
        const stride = 1;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            0.18518519, 0.22222222, 0.25925926, 0.40740741, 0.5, 0.59259259,
            0.62962963, 0.77777778, 0.92592593, 0.85185185, 1.05555556, 1.25925926,
            1.07407407, 1.33333333, 1.59259259, 1.2962963, 1.61111111, 1.92592593
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 4, 2, 3, 3] f=[2, 2, 2, 3, 3] s=1 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 4;
        const inHeight = 2;
        const inWidth = 3;
        const inChannels = 3;
        const outChannels = 3;
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            3.77199074, 3.85069444, 3.92939815, 4.2650463, 4.35763889, 4.45023148,
            6.73032407, 6.89236111, 7.05439815, 7.22337963, 7.39930556, 7.57523148,
            9.68865741, 9.93402778, 10.17939815, 10.18171296, 10.44097222, 10.70023148
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 5, 8, 7, 1] f=[1, 2, 3, 1, 1] s=[2, 3, 1] d=1 p=same', async () => {
        const batch = 1;
        const inDepth = 5;
        const inHeight = 8;
        const inWidth = 7;
        const inChannels = 1;
        const outChannels = 1;
        const fDepth = 1;
        const fHeight = 2;
        const fWidth = 3;
        const pad = 'same';
        const stride = [2, 3, 1];
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fDepth, fHeight, fWidth, pad, stride);
        const expectedOutput = [
            0.06071429, 0.08988095, 0.10238095, 0.11488095, 0.12738095, 0.13988095,
            0.08452381, 0.26071429, 0.35238095, 0.36488095, 0.37738095, 0.38988095,
            0.40238095, 0.23452381, 0.46071429, 0.61488095, 0.62738095, 0.63988095,
            0.65238095, 0.66488095, 0.38452381, 1.12738095, 1.48988095, 1.50238095,
            1.51488095, 1.52738095, 1.53988095, 0.88452381, 1.32738095, 1.75238095,
            1.76488095, 1.77738095, 1.78988095, 1.80238095, 1.03452381, 1.52738095,
            2.01488095, 2.02738095, 2.03988095, 2.05238095, 2.06488095, 1.18452381,
            2.19404762, 2.88988095, 2.90238095, 2.91488095, 2.92738095, 2.93988095,
            1.68452381, 2.39404762, 3.15238095, 3.16488095, 3.17738095, 3.18988095,
            3.20238095, 1.83452381, 2.59404762, 3.41488095, 3.42738095, 3.43988095,
            3.45238095, 3.46488095, 1.98452381
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 4, 2, 3, 3] f=[2, 2, 2, 3, 3] s=2 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 4;
        const inHeight = 2;
        const inWidth = 3;
        const inChannels = 3;
        const outChannels = 3;
        const fSize = 2;
        const pad = 'valid';
        const stride = 2;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            3.77199074, 3.85069444, 3.92939815, 9.68865741, 9.93402778, 10.17939815
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 6, 7, 8, 2] f=[3, 2, 1, 2, 3] s=3 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 6;
        const inHeight = 7;
        const inWidth = 8;
        const inChannels = 2;
        const outChannels = 3;
        const fDepth = 3;
        const fHeight = 2;
        const fWidth = 1;
        const pad = 'valid';
        const stride = 3;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fDepth, fHeight, fWidth, pad, stride);
        const expectedOutput = [
            1.51140873, 1.57167659, 1.63194444, 1.56349206, 1.62673611, 1.68998016,
            1.6155754, 1.68179563, 1.74801587, 1.9280754, 2.01215278, 2.09623016,
            1.98015873, 2.0672123, 2.15426587, 2.03224206, 2.12227183, 2.21230159,
            4.4280754, 4.65500992, 4.88194444, 4.48015873, 4.71006944, 4.93998016,
            4.53224206, 4.76512897, 4.99801587, 4.84474206, 5.09548611, 5.34623016,
            4.8968254, 5.15054563, 5.40426587, 4.94890873, 5.20560516, 5.46230159
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 4, 2, 3, 3] f=[2, 2, 2, 3, 3] s=2 d=1 p=same', async () => {
        const batch = 1;
        const inDepth = 4;
        const inHeight = 2;
        const inWidth = 3;
        const inChannels = 3;
        const outChannels = 3;
        const fSize = 2;
        const pad = 'same';
        const stride = 2;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            3.77199074, 3.85069444, 3.92939815, 2.0162037, 2.06597222, 2.11574074,
            9.68865741, 9.93402778, 10.17939815, 4.59953704, 4.73263889, 4.86574074
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 3, 3, 3, 1] f=[1, 1, 1, 1, 1] s=2 d=1 p=same', async () => {
        const batch = 1;
        const inDepth = 3;
        const inHeight = 3;
        const inWidth = 3;
        const inChannels = 1;
        const outChannels = 1;
        const fSize = 1;
        const pad = 'same';
        const stride = 2;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            0.03703704, 0.11111111, 0.25925926, 0.33333333, 0.7037037, 0.77777778,
            0.92592593, 1.
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 3, 3, 3, 1] f=[1, 1, 1, 1, 1] s=2 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 3;
        const inHeight = 3;
        const inWidth = 3;
        const inChannels = 1;
        const outChannels = 1;
        const fSize = 1;
        const pad = 'valid';
        const stride = 2;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            0.03703704, 0.11111111, 0.25925926, 0.33333333, 0.7037037, 0.77777778,
            0.92592593, 1.
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 7, 7, 7, 1] f=[2, 2, 2, 1, 1] s=3 d=1 p=same', async () => {
        const batch = 1;
        const inDepth = 7;
        const inHeight = 7;
        const inWidth = 7;
        const inChannels = 1;
        const outChannels = 1;
        const fSize = 2;
        const pad = 'same';
        const stride = 3;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            0.54081633, 0.58017493, 0.28061224, 0.81632653, 0.85568513, 0.40306122,
            0.41873178, 0.4340379, 0.19642857, 2.46938776, 2.50874636, 1.1377551,
            2.74489796, 2.78425656, 1.26020408, 1.16873178, 1.1840379, 0.51785714,
            1.09511662, 1.10604956, 0.44642857, 1.17164723, 1.18258017, 0.47704082,
            0.3691691, 0.37244898, 0.125
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 7, 7, 7, 1] f=[2, 2, 2, 1, 1] s=3 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 7;
        const inHeight = 7;
        const inWidth = 7;
        const inChannels = 1;
        const outChannels = 1;
        const fSize = 2;
        const pad = 'valid';
        const stride = 3;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fSize, fSize, fSize, pad, stride);
        const expectedOutput = [
            0.540816, 0.580175, 0.816327, 0.855685, 2.469388, 2.508746, 2.744898,
            2.784257
        ];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('x=[1, 2, 1, 2, 1] f=[2, 1, 2, 1, 2] s=1 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 2;
        const inHeight = 1;
        const inWidth = 2;
        const inChannels = 1;
        const outChannels = 2;
        const fDepth = 2;
        const fHeight = 1;
        const fWidth = 2;
        const pad = 'valid';
        const stride = 1;
        const result = runConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fDepth, fHeight, fWidth, pad, stride);
        const expectedOutput = [1.5625, 1.875];
        expectArraysClose(await result.data(), expectedOutput);
    });
    it('gradient with clones, x=[1,3,6,1,1] filter=[2,2,1,1,1] s=1 d=1 p=valid', async () => {
        const batch = 1;
        const inDepth = 3;
        const inHeight = 6;
        const inWidth = 1;
        const inChannels = 1;
        const outChannels = 1;
        const fDepth = 2;
        const fHeight = 2;
        const fWidth = 1;
        const pad = 'valid';
        const stride = 1;
        const [dx, dfilter] = runGradientConv3DTestCase(batch, inDepth, inHeight, inWidth, inChannels, outChannels, fDepth, fHeight, fWidth, pad, stride);
        const expectedFilterOutput = [60.0, 70.0, 120.0, 130.0];
        const expectedOutput = [
            1.0, 3.0, 3.0, 3.0, 3.0, 2.0, 4.0, 10.0, 10.0, 10.0, 10.0, 6.0, 3.0,
            7.0, 7.0, 7.0, 7.0, 4.0
        ];
        expectArraysClose(await dx.data(), expectedOutput);
        expectArraysClose(await dfilter.data(), expectedFilterOutput);
    });
    it('throws when passed x as a non-tensor', () => {
        const inputDepth = 1;
        const outputDepth = 1;
        const fSize = 1;
        const pad = 'valid';
        const stride = 1;
        const w = tf.tensor5d([2], [fSize, fSize, fSize, inputDepth, outputDepth]);
        expect(() => tf.conv3d({}, w, stride, pad))
            .toThrowError(/Argument 'x' passed to 'conv3d' must be a Tensor/);
    });
    it('throws when passed filter as a non-tensor', () => {
        const inputDepth = 1;
        const inputShape = [2, 2, 1, inputDepth];
        const pad = 'valid';
        const stride = 1;
        const x = tf.tensor4d([1, 2, 3, 4], inputShape);
        expect(() => tf.conv3d(x, {}, stride, pad))
            .toThrowError(/Argument 'filter' passed to 'conv3d' must be a Tensor/);
    });
    it('accepts a tensor-like object', async () => {
        const pad = 'valid';
        const stride = 1;
        const x = [[[[1], [2]], [[3], [4]]]]; // 2x2x1x1
        const w = [[[[[2]]]]]; // 1x1x1x1x1
        const result = tf.conv3d(x, w, stride, pad);
        expectArraysClose(await result.data(), [2, 4, 6, 8]);
    });
    it('throws when data format not NDHWC', () => {
        const inputDepth = 1;
        const outputDepth = 1;
        const inputShape = [2, 2, 1, inputDepth];
        const pad = 'valid';
        const fSize = 1;
        const stride = 1;
        const dataFormat = 'NCDHW';
        const x = tf.tensor4d([1, 2, 3, 4], inputShape);
        const w = tf.tensor5d([2], [fSize, fSize, fSize, inputDepth, outputDepth]);
        expect(() => tf.conv3d(x, w, stride, pad, dataFormat)).toThrowError();
    });
});
//# sourceMappingURL=data:application/json;base64,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