/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { ENGINE } from '../engine';
import { Conv3DBackpropInputV2 } from '../kernel_names';
import * as util from '../util';
import { op } from './operation';
import { reshape } from './reshape';
/**
 * Computes the derivative of the input of a 3D convolution.
 *
 * @param xShape The shape of the input: [batch, depth, height, width,
 * in_channels]. If length of 4, batch of 1 is assumed.
 * @param dy The derivative of the output, of rank 5 or rank 4 of shape
 *   `[batch, outDepth, outHeight, outWidth, in_channels]`.
 * If rank 4, batch of 1 is assumed.
 * @param filter The filter, rank 5, of shape
 *     `[filterDepth, filterHeight, filterWidth, inDepth, outDepth]`.
 * @param strides The strides of the convolution: `[strideDepth, strideHeight,
 * strideWidth]`.
 * @param pad The type of padding algorithm used:
 *    - `same` and stride 1: output will be of same size as input,
 *       regardless of filter size.
 *    - `valid`: output will be smaller than input if filter is larger
 *       than 1x1.
 */
function conv3DBackpropInput_(xShape, dy, filter, strides, pad) {
    util.assert(xShape.length === dy.rank, () => `Length of inShape ` +
        `(${xShape.length}) and rank of dy (${dy.rank}) must match`);
    let xShape5D = xShape;
    let dy5D = dy;
    let reshapedTo5D = false;
    if (dy.rank === 4) {
        reshapedTo5D = true;
        dy5D = reshape(dy, [1, dy.shape[0], dy.shape[1], dy.shape[2], dy.shape[3]]);
        xShape5D = [1, xShape[0], xShape[1], xShape[2], xShape[3]];
    }
    const inDepth = xShape5D[4];
    const outDepth = dy5D.shape[4];
    util.assert(xShape5D.length === 5, () => `Error in conv3dDerInput: inShape must be length 5, but got length ` +
        `${xShape5D.length}.`);
    util.assert(dy5D.rank === 5, () => `Error in conv3dDerInput: dy must be rank 5, but got ` +
        `rank ${dy5D.rank}`);
    util.assert(filter.rank === 5, () => `Error in conv3dDerInput: filter must be rank 5, but got ` +
        `rank ${filter.rank}`);
    util.assert(inDepth === filter.shape[3], () => `Error in conv3dDerInput: depth of input (${inDepth}) must ` +
        `match input depth for filter ${filter.shape[3]}.`);
    util.assert(outDepth === filter.shape[4], () => `Error in conv3dDerInput: depth of output (${outDepth}) must ` +
        `match output depth for filter ${filter.shape[4]}.`);
    const inputs = { dy: dy5D, filter };
    const attrs = { pad, strides, inputShape: xShape5D };
    // tslint:disable-next-line: no-unnecessary-type-assertion
    const res = ENGINE.runKernel(Conv3DBackpropInputV2, inputs, attrs);
    if (reshapedTo5D) {
        return reshape(res, [res.shape[1], res.shape[2], res.shape[3], res.shape[4]]);
    }
    return res;
}
export const conv3DBackpropInput = op({ conv3DBackpropInput_ });
//# sourceMappingURL=data:application/json;base64,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