/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
describeWithFlags('separableConv2d', ALL_ENVS, () => {
    it('input=1x3x3x1,f=2,s=1,d=1,p=valid,chMul=1,outDepth=2', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const depthwiseFilter = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2], [1, 1, inDepth * chMul, outDepth]);
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.10702161, -0.21404321, 0.10316753, -0.20633507, 0.06704096, -0.13408193,
            0.07788632, -0.15577264
        ]);
        expect(result.shape).toEqual([1, 2, 2, outDepth]);
    });
    it('input=1x3x3x1,f=2,s=1,d=1,p=valid,chMul=1,outDepth=2 in tensor', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const depthwiseFilter = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2], [1, 1, inDepth * chMul, outDepth]);
        const result = x.separableConv2d(depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.10702161, -0.21404321, 0.10316753, -0.20633507, 0.06704096,
            -0.13408193, 0.07788632, -0.15577264
        ]);
        expect(result.shape).toEqual([1, 2, 2, outDepth]);
    });
    it('input=1x3x3x1,f=2,s=1,d=1,p=valid,chMul=2,outDepth=2', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 2;
        const inDepth = 1;
        const outDepth = 3;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const depthwiseFilter = tf.tensor4d([
            0.303873, 0.229223, 0.144333, 0.803373, -0.303873, -0.229223,
            -0.144333, -0.803373
        ], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2, -0.1, 0.2, 0.15, 0.15], [1, 1, inDepth * chMul, outDepth]);
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.00305368, 0.0140969, 0.00980358, -0.10853045, -0.06339455, -0.0699412,
            0.11010849, 0.0347524, 0.05214475, 0.10307151, 0.02221644, 0.04224815
        ]);
        expect(result.shape).toEqual([1, 2, 2, outDepth]);
    });
    it('input=1x3x3x1,f=2,s=1,d=1,p=valid,chMul=1,outDepth=2,3D input', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const x = tf.tensor3d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [3, 3, inDepth]);
        const depthwiseFilter = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2], [1, 1, inDepth * chMul, outDepth]);
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.10702161, -0.21404321, 0.10316753, -0.20633507, 0.06704096,
            -0.13408193, 0.07788632, -0.15577264
        ]);
        expect(result.shape).toEqual([2, 2, outDepth]);
    });
    it('input=1x4x4x1,f=2,s=2,d=1,p=valid,chMul=1,outDepth=2', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = [2, 2];
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const x = tf.tensor4d([
            0.675707, 0.758567, 0.413529, 0.963967, 0.217291, 0.101335, 0.804231,
            0.329673, 0.924503, 0.728742, 0.180217, 0.210459, 0.133869, 0.650827,
            0.047613, 0.554795
        ], [1, 4, 4, inDepth]);
        const depthwiseFilter = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2], [1, 1, inDepth * chMul, outDepth]);
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.04919822, -0.09839644, 0.07275512, -0.14551024, 0.09901544, -0.19803089,
            0.05555845, -0.11111691
        ]);
        expect(result.shape).toEqual([1, 2, 2, outDepth]);
    });
    it('input=2x4x4x1,f=2,s=2,d=1,p=valid,chMul=1,outDepth=2', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = [2, 2];
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const x = tf.tensor4d([
            0.675707, 0.758567, 0.413529, 0.963967, 0.217291, 0.101335,
            0.804231, 0.329673, 0.924503, 0.728742, 0.180217, 0.210459,
            0.133869, 0.650827, 0.047613, 0.554795, -0.675707, -0.758567,
            -0.413529, -0.963967, -0.217291, -0.101335, -0.804231, -0.329673,
            -0.924503, -0.728742, -0.180217, -0.210459, -0.133869, -0.650827,
            -0.047613, -0.554795
        ], [2, 4, 4, inDepth]);
        const depthwiseFilter = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2], [1, 1, inDepth * chMul, outDepth]);
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.04919822, -0.09839644, 0.07275512, -0.14551024, 0.09901544, -0.19803089,
            0.05555845, -0.11111691, -0.04919822, 0.09839644, -0.07275512, 0.14551024,
            -0.09901544, 0.19803089, -0.05555845, 0.11111691
        ]);
        expect(result.shape).toEqual([2, 2, 2, outDepth]);
    });
    it('input=1x4x4x2,f=2,s=2,d=1,p=valid,chMul=1,outDepth=2', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = [2, 2];
        const chMul = 1;
        const inDepth = 2;
        const outDepth = 2;
        const x = tf.tensor4d([
            0.675707, 0.758567, 0.413529, 0.963967, 0.217291, 0.101335,
            0.804231, 0.329673, 0.924503, 0.728742, 0.180217, 0.210459,
            0.133869, 0.650827, 0.047613, 0.554795, -0.675707, -0.758567,
            -0.413529, -0.963967, -0.217291, -0.101335, -0.804231, -0.329673,
            -0.924503, -0.728742, -0.180217, -0.210459, -0.133869, -0.650827,
            -0.047613, -0.554795
        ], [1, 4, 4, inDepth]);
        const depthwiseFilter = tf.tensor4d([
            0.303873, 0.229223, 0.144333, 0.803373, 0.98976838, 0.56597068,
            0.42654137, 0.66445535
        ], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2, 0.05, -0.05], [1, 1, inDepth * chMul, outDepth]);
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.20072255, -0.32641545, 0.08474462, -0.11823604, -0.20072255, 0.32641545,
            -0.08474462, 0.11823604
        ]);
        expect(result.shape).toEqual([1, 2, 2, outDepth]);
    });
    it('input=1x4x4x1,f=2,s=1,d=2,p=valid,chMul=1,outDepth=2', async () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const dilationRate = 2;
        const x = tf.tensor4d([
            0.675707, 0.758567, 0.413529, 0.963967, 0.217291, 0.101335, 0.804231,
            0.329673, 0.924503, 0.728742, 0.180217, 0.210459, 0.133869, 0.650827,
            0.047613, 0.554795
        ], [1, 4, 4, inDepth]);
        const depthwiseFilter = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2], [1, 1, inDepth * chMul, outDepth]);
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad, dilationRate);
        expectArraysClose(await result.data(), [
            0.05783373, -0.11566745, 0.07257301, -0.14514601, 0.03079498, -0.06158997,
            0.06460048, -0.12920095
        ]);
        expect(result.shape).toEqual([1, 2, 2, outDepth]);
    });
    it('input=1x4x4x1,f=2,s=1,d=1,p=same,chMul=1,outDepth=2', async () => {
        const fSize = 2;
        const pad = 'same';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const x = tf.tensor4d([
            0.675707, 0.758567, 0.413529, 0.963967, 0.217291, 0.101335, 0.804231,
            0.329673, 0.924503, 0.728742, 0.180217, 0.210459, 0.133869, 0.650827,
            0.047613, 0.554795
        ], [1, 4, 4, inDepth]);
        const depthwiseFilter = tf.tensor4d([0.303873, 0.229223, 0.144333, 0.803373], [fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.tensor4d([0.1, -0.2], [1, 1, inDepth * chMul, outDepth]);
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.04919822, -0.09839644, 0.09860218, -0.19720435, 0.07275512, -0.14551024,
            0.03405062, -0.06810125, 0.08081452, -0.16162904, 0.04651042, -0.09302084,
            0.05150411, -0.10300821, 0.01305549, -0.02611098, 0.09901544, -0.19803089,
            0.03949417, -0.07898834, 0.05555845, -0.11111691, 0.0144028, -0.02880561,
            0.01898637, -0.03797274, 0.02086828, -0.04173655, 0.01416401, -0.02832802,
            0.01685872, -0.03371745
        ]);
        expect(result.shape).toEqual([1, 4, 4, outDepth]);
    });
    it('TensorLike', async () => {
        const pad = 'valid';
        const stride = 1;
        const outDepth = 2;
        const x = [[
                [[0.230664], [0.987388], [0.0685208]],
                [[0.419224], [0.887861], [0.731641]],
                [[0.0741907], [0.409265], [0.351377]]
            ]];
        const depthwiseFilter = [[[[0.303873]], [[0.229223]]], [[[0.144333]], [[0.803373]]]];
        const pointwiseFilter = [[[[0.1, -0.2]]]];
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.10702161, -0.21404321, 0.10316753, -0.20633507, 0.06704096, -0.13408193,
            0.07788632, -0.15577264
        ]);
        expect(result.shape).toEqual([1, 2, 2, outDepth]);
    });
    it('TensorLike Chained', async () => {
        const pad = 'valid';
        const stride = 1;
        const outDepth = 2;
        const inDepth = 1;
        const x = tf.tensor4d([
            0.230664, 0.987388, 0.0685208, 0.419224, 0.887861, 0.731641,
            0.0741907, 0.409265, 0.351377
        ], [1, 3, 3, inDepth]);
        const depthwiseFilter = [[[[0.303873]], [[0.229223]]], [[[0.144333]], [[0.803373]]]];
        const pointwiseFilter = [[[[0.1, -0.2]]]];
        const result = x.separableConv2d(depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.10702161, -0.21404321, 0.10316753, -0.20633507, 0.06704096, -0.13408193,
            0.07788632, -0.15577264
        ]);
        expect(result.shape).toEqual([1, 2, 2, outDepth]);
    });
    it('Incorrect input rank raises error', () => {
        // tslint:disable-next-line:no-any
        const x = tf.zeros([4, 4]);
        const depthwiseFilter = tf.zeros([2, 2, 1, 3]);
        const pointwiseFilter = tf.zeros([1, 1, 2, 4]);
        expect(() => tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, 1, 'valid'))
            .toThrowError(/rank 4/);
    });
    it('Incorrect depthwise filter rank raises error', () => {
        const x = tf.zeros([1, 4, 4, 1]);
        // tslint:disable-next-line:no-any
        const depthwiseFilter = tf.zeros([2, 2, 1]);
        const pointwiseFilter = tf.zeros([1, 1, 2, 4]);
        expect(() => tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, 1, 'valid'))
            .toThrowError(/rank 4/);
    });
    it('Incorrect depthwise filter rank raises error', () => {
        const x = tf.zeros([1, 4, 4, 1]);
        const depthwiseFilter = tf.zeros([2, 2, 1, 3]);
        // tslint:disable-next-line:no-any
        const pointwiseFilter = tf.zeros([1, 1, 2]);
        expect(() => tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, 1, 'valid'))
            .toThrowError(/rank 4/);
    });
    it('Incorrect point filter 1st dimension raises error', () => {
        const x = tf.zeros([1, 4, 4, 1]);
        const depthwiseFilter = tf.zeros([2, 2, 1, 3]);
        const pointwiseFilter = tf.zeros([2, 1, 3, 6]);
        expect(() => tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, 1, 'valid'))
            .toThrowError(/must be 1, but got 2/);
    });
    it('Incorrect point filter 2nd dimension raises error', () => {
        const x = tf.zeros([1, 4, 4, 1]);
        const depthwiseFilter = tf.zeros([2, 2, 1, 3]);
        const pointwiseFilter = tf.zeros([1, 5, 3, 6]);
        expect(() => tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, 1, 'valid'))
            .toThrowError(/must be 1, but got 5/);
    });
    it('Incorrect pointwise filter 3rd dimension raises error', () => {
        const x = tf.zeros([1, 4, 4, 1]);
        const depthwiseFilter = tf.zeros([2, 2, 1, 3]);
        const pointwiseFilter = tf.zeros([1, 1, 4, 6]);
        expect(() => tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, 1, 'valid'))
            .toThrowError(/must be 3, but got 4/);
    });
    it('throws when passed x as a non-tensor', () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const depthwiseFilter = tf.zeros([fSize, fSize, inDepth, chMul]);
        const pointwiseFilter = tf.zeros([1, 1, inDepth * chMul, outDepth]);
        const e = /Argument 'x' passed to 'separableConv2d' must be a Tensor/;
        expect(() => tf.separableConv2d({}, depthwiseFilter, pointwiseFilter, stride, pad))
            .toThrowError(e);
    });
    it('throws when passed depthwiseFilter as a non-tensor', () => {
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const outDepth = 2;
        const x = tf.zeros([1, 3, 3, inDepth]);
        const pointwiseFilter = tf.zeros([1, 1, inDepth * chMul, outDepth]);
        const e = new RegExp('Argument \'depthwiseFilter\' passed to \'separableConv2d\'' +
            ' must be a Tensor');
        expect(() => tf.separableConv2d(x, {}, pointwiseFilter, stride, pad))
            .toThrowError(e);
    });
    it('throws when passed pointwiseFilter as a non-tensor', () => {
        const fSize = 2;
        const pad = 'valid';
        const stride = 1;
        const chMul = 1;
        const inDepth = 1;
        const x = tf.zeros([1, 3, 3, inDepth]);
        const depthwiseFilter = tf.zeros([fSize, fSize, inDepth, chMul]);
        const e = new RegExp('Argument \'pointwiseFilter\' passed to \'separableConv2d\'' +
            ' must be a Tensor');
        expect(() => tf.separableConv2d(x, depthwiseFilter, {}, stride, pad))
            .toThrowError(e);
    });
    it('accepts a tensor-like object', async () => {
        const pad = 'valid';
        const stride = 1;
        const outDepth = 2;
        // 3x3x1
        const x = [
            [[0.230664], [0.987388], [0.0685208]],
            [[0.419224], [0.887861], [0.731641]],
            [[0.0741907], [0.409265], [0.351377]]
        ];
        // 2x2x1x1
        const depthwiseFilter = [[[[0.303873]], [[0.229223]]], [[[0.144333]], [[0.803373]]]];
        // 1x1x1x2
        const pointwiseFilter = [[[[0.1, -0.2]]]];
        const result = tf.separableConv2d(x, depthwiseFilter, pointwiseFilter, stride, pad);
        expectArraysClose(await result.data(), [
            0.10702161, -0.21404321, 0.10316753, -0.20633507, 0.06704096, -0.13408193,
            0.07788632, -0.15577264
        ]);
        expect(result.shape).toEqual([2, 2, outDepth]);
    });
});
//# sourceMappingURL=data:application/json;base64,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