/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose, expectArraysEqual } from '../test_util';
describeWithFlags('concat1d', ALL_ENVS, () => {
    it('3 + 5', async () => {
        const a = tf.tensor1d([3]);
        const b = tf.tensor1d([5]);
        const result = tf.concat1d([a, b]);
        const expected = [3, 5];
        expectArraysClose(await result.data(), expected);
    });
    it('TensorLike 3 + 5', async () => {
        const a = [3];
        const b = [5];
        const result = tf.concat1d([a, b]);
        const expected = [3, 5];
        expectArraysClose(await result.data(), expected);
    });
    it('TensorLike Chained 3 + 5', async () => {
        const a = tf.tensor1d([3]);
        const b = [5];
        const result = a.concat([b]);
        const expected = [3, 5];
        expectArraysClose(await result.data(), expected);
    });
    it('3 + [5,7]', async () => {
        const a = tf.tensor1d([3]);
        const b = tf.tensor1d([5, 7]);
        const result = tf.concat1d([a, b]);
        const expected = [3, 5, 7];
        expectArraysClose(await result.data(), expected);
    });
    it('[3,5] + 7', async () => {
        const a = tf.tensor1d([3, 5]);
        const b = tf.tensor1d([7]);
        const result = tf.concat1d([a, b]);
        const expected = [3, 5, 7];
        expectArraysClose(await result.data(), expected);
    });
    it('3 + 5 + 7 + 9', async () => {
        const a = tf.tensor1d([3]);
        const b = tf.tensor1d([5]);
        const c = tf.tensor1d([7]);
        const d = tf.tensor1d([9]);
        const result = tf.concat1d([a, b, c, d]);
        expectArraysClose(await result.data(), [3, 5, 7, 9]);
    });
    it('single tensor', async () => {
        const a = tf.tensor1d([3]);
        const result = tf.concat1d([a]);
        expectArraysClose(await result.data(), [3]);
    });
    it('accepts a tensor-like object', async () => {
        const a = [3];
        const b = [5];
        const result = tf.concat1d([a, b]);
        const expected = [3, 5];
        expectArraysClose(await result.data(), expected);
    });
    it('concat complex input', async () => {
        // [1+1j, 2+2j]
        const c1 = tf.complex([1, 2], [1, 2]);
        // [3+3j, 4+4j]
        const c2 = tf.complex([3, 4], [3, 4]);
        const axis = 0;
        const result = tf.concat([c1, c2], axis);
        const expected = [1, 1, 2, 2, 3, 3, 4, 4];
        expect(result.dtype).toEqual('complex64');
        expectArraysClose(await result.data(), expected);
    });
});
describeWithFlags('concat2d', ALL_ENVS, () => {
    it('[[3]] + [[5]], axis=0', async () => {
        const axis = 0;
        const a = tf.tensor2d([3], [1, 1]);
        const b = tf.tensor2d([5], [1, 1]);
        const result = tf.concat2d([a, b], axis);
        const expected = [3, 5];
        expect(result.shape).toEqual([2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('TensorLike [[3]] + [[5]], axis=0', async () => {
        const axis = 0;
        const a = [[3]];
        const b = [[5]];
        const result = tf.concat2d([a, b], axis);
        const expected = [3, 5];
        expect(result.shape).toEqual([2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('TensorLike Chained [[3]] + [[5]], axis=0', async () => {
        const axis = 0;
        const a = tf.tensor2d([3], [1, 1]);
        const b = [[5]];
        const result = a.concat([b], axis);
        const expected = [3, 5];
        expect(result.shape).toEqual([2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('[[3]] + [[5]], axis=1', async () => {
        const axis = 1;
        const a = tf.tensor2d([3], [1, 1]);
        const b = tf.tensor2d([5], [1, 1]);
        const result = tf.concat2d([a, b], axis);
        const expected = [3, 5];
        expect(result.shape).toEqual([1, 2]);
        expectArraysClose(await result.data(), expected);
    });
    it('[[1, 2], [3, 4]] + [[5, 6]], axis=0', async () => {
        const axis = 0;
        const a = tf.tensor2d([[1, 2], [3, 4]], [2, 2]);
        const b = tf.tensor2d([[5, 6]], [1, 2]);
        const result = tf.concat2d([a, b], axis);
        const expected = [1, 2, 3, 4, 5, 6];
        expect(result.shape).toEqual([3, 2]);
        expectArraysClose(await result.data(), expected);
    });
    it('[[1, 2],[3, 4]] + [[5, 6]] + [[7, 8]], axis=0', async () => {
        const axis = 0;
        const a = tf.tensor2d([[1, 2], [3, 4]]);
        const b = tf.tensor2d([[5, 6]]);
        const c = tf.tensor2d([[7, 8]]);
        const result = tf.concat2d([a, b, c], axis);
        const expected = [1, 2, 3, 4, 5, 6, 7, 8];
        expect(result.shape).toEqual([4, 2]);
        expectArraysClose(await result.data(), expected);
    });
    it('[[1, 2], [3, 4]] + [[5, 6]], axis=1 throws error', () => {
        const axis = 1;
        const a = tf.tensor2d([[1, 2], [3, 4]], [2, 2]);
        const b = tf.tensor2d([[5, 6]], [1, 2]);
        expect(() => tf.concat2d([a, b], axis)).toThrowError();
    });
    it('[[1, 2], [3, 4]] + [[5, 6], [7, 8]], axis=1', async () => {
        const axis = 1;
        const a = tf.tensor2d([[1, 2], [3, 4]], [2, 2]);
        const b = tf.tensor2d([[5, 6], [7, 8]], [2, 2]);
        const result = tf.concat2d([a, b], axis);
        const expected = [1, 2, 5, 6, 3, 4, 7, 8];
        expect(result.shape).toEqual([2, 4]);
        expectArraysClose(await result.data(), expected);
    });
    it('[[1, 2],[3, 4]] + [[5, 6],[7, 8]] + [[9, 10],[11, 12]], axis=1', async () => {
        const axis = 1;
        const a = tf.tensor2d([[1, 2], [3, 4]]);
        const b = tf.tensor2d([[5, 6], [7, 8]]);
        const c = tf.tensor2d([[9, 10], [11, 12]]);
        const result = tf.concat2d([a, b, c], axis);
        const expected = [1, 2, 5, 6, 9, 10, 3, 4, 7, 8, 11, 12];
        expect(result.shape).toEqual([2, 6]);
        expectArraysClose(await result.data(), expected);
    });
    it('accepts a tensor-like object', async () => {
        const axis = 0;
        const a = [[3]];
        const b = [[5]];
        const result = tf.concat2d([a, b], axis);
        const expected = [3, 5];
        expect(result.shape).toEqual([2, 1]);
        expectArraysClose(await result.data(), expected);
    });
    it('concat zero-sized tensors', async () => {
        const a = tf.tensor2d([], [0, 5]);
        const b = tf.tensor2d([], [0, 5]);
        const c = tf.tensor2d([], [0, 5]);
        const res = tf.concat([a, b, c], /* axis */ 0);
        expect(res.shape).toEqual([0, 5]);
        expectArraysEqual(await res.data(), []);
        const res2 = tf.concat([a, b, c], /* axis */ 1);
        expect(res2.shape).toEqual([0, 15]);
        expectArraysEqual(await res2.data(), []);
    });
    it('concat complex input axis=0', async () => {
        // [[1+1j, 2+2j], [3+3j, 4+4j]]
        const c1 = tf.complex([[1, 2], [3, 4]], [[1, 2], [3, 4]]);
        // [[5+5j, 6+6j], [7+7j, 8+8j]]
        const c2 = tf.complex([[5, 6], [7, 8]], [[5, 6], [7, 8]]);
        const axis = 0;
        const result = tf.concat([c1, c2], axis);
        const expected = [1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8];
        expect(result.dtype).toEqual('complex64');
        expectArraysClose(await result.data(), expected);
    });
    it('concat complex input axis=1', async () => {
        // [[1+1j, 2+2j], [3+3j, 4+4j]]
        const c1 = tf.complex([[1, 2], [3, 4]], [[1, 2], [3, 4]]);
        // [[5+5j, 6+6j], [7+7j, 8+8j]]
        const c2 = tf.complex([[5, 6], [7, 8]], [[5, 6], [7, 8]]);
        const axis = 1;
        const result = tf.concat([c1, c2], axis);
        const expected = [1, 1, 2, 2, 5, 5, 6, 6, 3, 3, 4, 4, 7, 7, 8, 8];
        expect(result.dtype).toEqual('complex64');
        expectArraysClose(await result.data(), expected);
    });
});
describeWithFlags('concat3d', ALL_ENVS, () => {
    beforeAll(() => {
        jasmine.DEFAULT_TIMEOUT_INTERVAL = 1000000;
    });
    it('shapes correct concat axis=-1', async () => {
        const tensor1 = tf.tensor3d([1, 2, 3], [1, 1, 3]);
        const tensor2 = tf.tensor3d([4, 5, 6], [1, 1, 3]);
        const values = tf.concat3d([tensor1, tensor2], -1);
        expect(values.shape).toEqual([1, 1, 6]);
        expectArraysClose(await values.data(), [1, 2, 3, 4, 5, 6]);
    });
    it('shapes correct concat axis=0', async () => {
        const tensor1 = tf.tensor3d([1, 2, 3], [1, 1, 3]);
        const tensor2 = tf.tensor3d([4, 5, 6], [1, 1, 3]);
        const values = tf.concat3d([tensor1, tensor2], 0);
        expect(values.shape).toEqual([2, 1, 3]);
        expectArraysClose(await values.data(), [1, 2, 3, 4, 5, 6]);
    });
    it('concat axis=0', async () => {
        const tensor1 = tf.tensor3d([1, 11, 111, 2, 22, 222], [1, 2, 3]);
        const tensor2 = tf.tensor3d([5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888], [2, 2, 3]);
        const values = tf.concat3d([tensor1, tensor2], 0);
        expect(values.shape).toEqual([3, 2, 3]);
        expectArraysClose(await values.data(), [
            1, 11, 111, 2, 22, 222, 5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888
        ]);
    });
    it('TensorLike concat axis=0', async () => {
        const tensor1 = [[[1, 11, 111], [2, 22, 222]]];
        const tensor2 = [[[5, 55, 555], [6, 66, 666]], [[7, 77, 777], [8, 88, 888]]];
        const values = tf.concat3d([tensor1, tensor2], 0);
        expect(values.shape).toEqual([3, 2, 3]);
        expectArraysClose(await values.data(), [
            1, 11, 111, 2, 22, 222, 5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888
        ]);
    });
    it('Accepts string tensor.', async () => {
        const tensor1 = tf.tensor3d(['one', 'two', 'three'], [1, 1, 3], 'string');
        const tensor2 = tf.tensor3d(['four', 'five', 'six'], [1, 1, 3], 'string');
        const values = tf.concat3d([tensor1, tensor2], 0);
        expect(values.shape).toEqual([2, 1, 3]);
        expectArraysClose(await values.data(), ['one', 'two', 'three', 'four', 'five', 'six']);
    });
    it('TensorLike Chained concat axis=0', async () => {
        const tensor1 = tf.tensor3d([1, 11, 111, 2, 22, 222], [1, 2, 3]);
        const tensor2 = [[[5, 55, 555], [6, 66, 666]], [[7, 77, 777], [8, 88, 888]]];
        const values = tensor1.concat([tensor2], 0);
        expect(values.shape).toEqual([3, 2, 3]);
        expectArraysClose(await values.data(), [
            1, 11, 111, 2, 22, 222, 5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888
        ]);
    });
    it('shapes correct concat axis=1', async () => {
        const tensor1 = tf.tensor3d([1, 2, 3], [1, 1, 3]);
        const tensor2 = tf.tensor3d([4, 5, 6], [1, 1, 3]);
        const values = tf.concat3d([tensor1, tensor2], 1);
        expect(values.shape).toEqual([1, 2, 3]);
        expectArraysClose(await values.data(), [1, 2, 3, 4, 5, 6]);
    });
    it('concat axis=1', async () => {
        const tensor1 = tf.tensor3d([1, 11, 111, 3, 33, 333], [2, 1, 3]);
        const tensor2 = tf.tensor3d([5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888], [2, 2, 3]);
        const values = tf.concat3d([tensor1, tensor2], 1);
        expect(values.shape).toEqual([2, 3, 3]);
        expectArraysClose(await values.data(), [
            1, 11, 111, 5, 55, 555, 6, 66, 666, 3, 33, 333, 7, 77, 777, 8, 88, 888
        ]);
    });
    it('shapes correct concat axis=2', async () => {
        const tensor1 = tf.tensor3d([1, 2, 3], [1, 1, 3]);
        const tensor2 = tf.tensor3d([4, 5, 6], [1, 1, 3]);
        const values = tf.concat3d([tensor1, tensor2], 2);
        expect(values.shape).toEqual([1, 1, 6]);
        expectArraysClose(await values.data(), [1, 2, 3, 4, 5, 6]);
    });
    it('concat a large number of tensors, axis=0', async () => {
        const tensors = [];
        const expected = [];
        for (let i = 0; i < 100; i++) {
            tensors.push(tf.tensor([i], [1]));
            expected.push(i);
        }
        const axis = 0;
        const res = tf.concat(tensors, axis);
        expect(res.shape).toEqual([100]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), expected);
    });
    it('concat a large number of tensors, axis=1', async () => {
        const tensors = [];
        const expected = [];
        for (let i = 0; i < 100; i++) {
            tensors.push(tf.tensor([i], [1, 1]));
            expected.push(i);
        }
        const axis = 1;
        const res = tf.concat(tensors, axis);
        expect(res.shape).toEqual([1, 100]);
        expect(res.dtype).toBe('float32');
        expectArraysClose(await res.data(), expected);
    });
    it('concat axis=2', async () => {
        const tensor1 = tf.tensor3d([1, 11, 2, 22, 3, 33, 4, 44], [2, 2, 2]);
        const tensor2 = tf.tensor3d([5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888], [2, 2, 3]);
        const values = tf.concat3d([tensor1, tensor2], 2);
        expect(values.shape).toEqual([2, 2, 5]);
        expectArraysClose(await values.data(), [
            1, 11, 5, 55, 555, 2, 22, 6, 66, 666,
            3, 33, 7, 77, 777, 4, 44, 8, 88, 888
        ]);
    });
    it('concat throws when invalid non-axis shapes, axis=0', () => {
        const axis = 0;
        const x1 = tf.tensor3d([1, 11, 111], [1, 1, 3]);
        const x2 = tf.tensor3d([5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888], [2, 2, 3]);
        expect(() => tf.concat3d([x1, x2], axis)).toThrowError();
    });
    it('concat throws when invalid non-axis shapes, axis=1', () => {
        const axis = 1;
        const x1 = tf.tensor3d([1, 11, 111], [1, 1, 3]);
        const x2 = tf.tensor3d([5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888], [2, 2, 3]);
        expect(() => tf.concat3d([x1, x2], axis)).toThrowError();
    });
    it('concat throws when invalid non-axis shapes, axis=2', () => {
        const axis = 2;
        const x1 = tf.tensor3d([1, 11, 2, 22], [1, 2, 2]);
        const x2 = tf.tensor3d([5, 55, 555, 6, 66, 666, 7, 77, 777, 8, 88, 888], [2, 2, 3]);
        expect(() => tf.concat3d([x1, x2], axis)).toThrowError();
    });
    it('gradient concat axis=0', async () => {
        const x1 = tf.tensor3d([1, 11, 2, 22], [1, 2, 2]);
        const x2 = tf.tensor3d([5, 55, 6, 66, 7, 77, 8, 88], [2, 2, 2]);
        const dy = tf.tensor3d([66, 6, 55, 5, 44, 4, 33, 3, 22, 2, 11, 1], [3, 2, 2]);
        const axis = 0;
        const grads = tf.grads((x1, x2) => tf.concat3d([x1, x2], axis));
        const [dx1, dx2] = grads([x1, x2], dy);
        expect(dx1.shape).toEqual(x1.shape);
        expectArraysClose(await dx1.data(), [66, 6, 55, 5]);
        expect(dx2.shape).toEqual(x2.shape);
        expectArraysClose(await dx2.data(), [44, 4, 33, 3, 22, 2, 11, 1]);
    });
    it('gradient with clones', async () => {
        const x1 = tf.tensor3d([1, 11, 2, 22], [1, 2, 2]);
        const x2 = tf.tensor3d([5, 55, 6, 66, 7, 77, 8, 88], [2, 2, 2]);
        const dy = tf.tensor3d([66, 6, 55, 5, 44, 4, 33, 3, 22, 2, 11, 1], [3, 2, 2]);
        const axis = 0;
        const grads = tf.grads((x1, x2) => tf.concat3d([x1.clone(), x2.clone()], axis).clone());
        const [dx1, dx2] = grads([x1, x2], dy);
        expect(dx1.shape).toEqual(x1.shape);
        expectArraysClose(await dx1.data(), [66, 6, 55, 5]);
        expect(dx2.shape).toEqual(x2.shape);
        expectArraysClose(await dx2.data(), [44, 4, 33, 3, 22, 2, 11, 1]);
    });
    it('gradient concat axis=1', async () => {
        const x1 = tf.tensor3d([1, 11, 2, 22], [2, 1, 2]);
        const x2 = tf.tensor3d([3, 33, 4, 44, 5, 55, 6, 66], [2, 2, 2]);
        const dy = tf.tensor3d([66, 6, 55, 5, 44, 4, 33, 3, 22, 2, 11, 1], [2, 3, 2]);
        const axis = 1;
        const grads = tf.grads((x1, x2) => tf.concat3d([x1, x2], axis));
        const [dx1, dx2] = grads([x1, x2], dy);
        expect(dx1.shape).toEqual(x1.shape);
        expectArraysClose(await dx1.data(), [66, 6, 33, 3]);
        expect(dx2.shape).toEqual(x2.shape);
        expectArraysClose(await dx2.data(), [55, 5, 44, 4, 22, 2, 11, 1]);
    });
    it('gradient concat axis=2', async () => {
        const x1 = tf.tensor3d([1, 2, 3, 4], [2, 2, 1]);
        const x2 = tf.tensor3d([5, 55, 6, 66, 7, 77, 8, 88], [2, 2, 2]);
        const dy = tf.tensor3d([4, 40, 400, 3, 30, 300, 2, 20, 200, 1, 10, 100], [2, 2, 3]);
        const axis = 2;
        const grads = tf.grads((x1, x2) => tf.concat3d([x1, x2], axis));
        const [dx1, dx2] = grads([x1, x2], dy);
        expect(dx1.shape).toEqual(x1.shape);
        expectArraysClose(await dx1.data(), [4, 3, 2, 1]);
        expect(dx2.shape).toEqual(x2.shape);
        expectArraysClose(await dx2.data(), [40, 400, 30, 300, 20, 200, 10, 100]);
    });
    it('gradient concat axis=-1', async () => {
        const x1 = tf.tensor3d([1, 2, 3, 4], [2, 2, 1]);
        const x2 = tf.tensor3d([5, 55, 6, 66, 7, 77, 8, 88], [2, 2, 2]);
        const dy = tf.tensor3d([4, 40, 400, 3, 30, 300, 2, 20, 200, 1, 10, 100], [2, 2, 3]);
        const axis = -1;
        const grads = tf.grads((x1, x2) => tf.concat3d([x1, x2], axis));
        const [dx1, dx2] = grads([x1, x2], dy);
        expect(dx1.shape).toEqual(x1.shape);
        expectArraysClose(await dx1.data(), [4, 3, 2, 1]);
        expect(dx2.shape).toEqual(x2.shape);
        expectArraysClose(await dx2.data(), [40, 400, 30, 300, 20, 200, 10, 100]);
    });
    it('accepts a tensor-like object', async () => {
        const tensor1 = [[[1, 2, 3]]]; // 1x1x3
        const tensor2 = [[[4, 5, 6]]]; // 1x1x3
        const values = tf.concat3d([tensor1, tensor2], 0);
        expect(values.shape).toEqual([2, 1, 3]);
        expectArraysClose(await values.data(), [1, 2, 3, 4, 5, 6]);
    });
    it('concat tensors with 0 in their shape', async () => {
        const tensor1 = tf.tensor3d([1, 2, 3, 4, 5, 6], [2, 3, 1]);
        const tensor2 = tf.tensor3d([], [0, 3, 1]);
        const values = tf.concat3d([tensor1, tensor2], 0);
        expect(values.shape).toEqual([2, 3, 1]);
        expectArraysClose(await values.data(), [1, 2, 3, 4, 5, 6]);
    });
    it('concat complex input axis=0', async () => {
        // [[[1+1j, 2+2j], [3+3j, 4+4j], [5+5j, 6+6j]]]
        const c1 = tf.complex([[[1, 2], [3, 4], [5, 6]]], [[[1, 2], [3, 4], [5, 6]]]);
        // [[[7+7j, 8+8j], [9+9j, 10+10j], [11+11j, 12+12j]]]
        const c2 = tf.complex([[[7, 8], [9, 10], [11, 12]]], [[[7, 8], [9, 10], [11, 12]]]);
        const axis = 0;
        const result = tf.concat([c1, c2], axis);
        const expected = [
            1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6,
            7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12
        ];
        expect(result.dtype).toEqual('complex64');
        expectArraysClose(await result.data(), expected);
    });
    it('concat complex input axis=1', async () => {
        // [[[1+1j, 2+2j], [3+3j, 4+4j], [5+5j, 6+6j]]]
        const c1 = tf.complex([[[1, 2], [3, 4], [5, 6]]], [[[1, 2], [3, 4], [5, 6]]]);
        // [[[7+7j, 8+8j], [9+9j, 10+10j], [11+11j, 12+12j]]]
        const c2 = tf.complex([[[7, 8], [9, 10], [11, 12]]], [[[7, 8], [9, 10], [11, 12]]]);
        const axis = 1;
        const result = tf.concat([c1, c2], axis);
        const expected = [
            1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6,
            7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12
        ];
        expect(result.dtype).toEqual('complex64');
        expectArraysClose(await result.data(), expected);
    });
    it('concat complex input axis=1', async () => {
        // [[[1+1j, 2+2j], [3+3j, 4+4j], [5+5j, 6+6j]]]
        const c1 = tf.complex([[[1, 2], [3, 4], [5, 6]]], [[[1, 2], [3, 4], [5, 6]]]);
        // [[[7+7j, 8+8j], [9+9j, 10+10j], [11+11j, 12+12j]]]
        const c2 = tf.complex([[[7, 8], [9, 10], [11, 12]]], [[[7, 8], [9, 10], [11, 12]]]);
        const axis = 2;
        const result = tf.concat([c1, c2], axis);
        const expected = [
            1, 1, 2, 2, 7, 7, 8, 8, 3, 3, 4, 4,
            9, 9, 10, 10, 5, 5, 6, 6, 11, 11, 12, 12
        ];
        expect(result.dtype).toEqual('complex64');
        expectArraysClose(await result.data(), expected);
    });
});
describeWithFlags('concat throws for non-tensors', ALL_ENVS, () => {
    it('throws when passed a non-tensor', () => {
        expect(() => tf.concat([{}]))
            .toThrowError(/Argument 'tensors\[0\]' passed to 'concat' must be a Tensor/);
    });
    it('accepts a tensor-like object', async () => {
        const tensor1 = [[[1, 2, 3, 4]]]; // 1x1x4
        const tensor2 = [[[4, 5, 6, 7]]]; // 1x1x4
        const values = tf.concat([tensor1, tensor2], 0);
        expect(values.shape).toEqual([2, 1, 4]);
        expectArraysClose(await values.data(), [1, 2, 3, 4, 4, 5, 6, 7]);
    });
});
describeWithFlags('memory test', ALL_ENVS, () => {
    it('returns a new tensor when op is effectively a no-op.', async () => {
        const a = tf.tensor1d([]);
        const b = tf.tensor1d([3]);
        const result = tf.concat([a, b]);
        a.dispose();
        b.dispose();
        expectArraysClose(await result.data(), [3]);
    });
    it('ensure no memory leak', async () => {
        const numTensorsBefore = tf.memory().numTensors;
        const numDataIdBefore = tf.engine().backend.numDataIds();
        const a = tf.tensor1d([]);
        const b = tf.tensor1d([3]);
        const result = tf.concat([a, b]);
        a.dispose();
        b.dispose();
        result.dispose();
        const numTensorsAfter = tf.memory().numTensors;
        const numDataIdAfter = tf.engine().backend.numDataIds();
        expect(numTensorsAfter).toBe(numTensorsBefore);
        expect(numDataIdAfter).toBe(numDataIdBefore);
    });
});
//# sourceMappingURL=data:application/json;base64,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