/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
describeWithFlags('batchToSpaceND', ALL_ENVS, () => {
    it('tensor4d, input shape=[4, 1, 1, 1], blockShape=[2, 2]', async () => {
        const t = tf.tensor4d([1, 2, 3, 4], [4, 1, 1, 1]);
        const blockShape = [2, 2];
        const crops = [[0, 0], [0, 0]];
        const res = tf.batchToSpaceND(t, blockShape, crops);
        expect(res.shape).toEqual([1, 2, 2, 1]);
        expectArraysClose(await res.data(), [1, 2, 3, 4]);
    });
    it('tensor4d, input shape=[4, 1, 1, 3], blockShape=[2, 2]', async () => {
        const t = tf.tensor4d([1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12], [4, 1, 1, 3]);
        const blockShape = [2, 2];
        const crops = [[0, 0], [0, 0]];
        const res = tf.batchToSpaceND(t, blockShape, crops);
        expect(res.shape).toEqual([1, 2, 2, 3]);
        expectArraysClose(await res.data(), [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]);
    });
    it('tensor4d, input shape=[4, 2, 2, 1], blockShape=[2, 2]', async () => {
        const t = tf.tensor4d([1, 3, 9, 11, 2, 4, 10, 12, 5, 7, 13, 15, 6, 8, 14, 16], [4, 2, 2, 1]);
        const blockShape = [2, 2];
        const crops = [[0, 0], [0, 0]];
        const res = tf.batchToSpaceND(t, blockShape, crops);
        expect(res.shape).toEqual([1, 4, 4, 1]);
        expectArraysClose(await res.data(), [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16]);
    });
    it('tensor4d, input shape=[8, 1, 3, 1], blockShape=[2, 2]', async () => {
        const t = tf.tensor4d([
            0, 1, 3, 0, 9, 11, 0, 2, 4, 0, 10, 12,
            0, 5, 7, 0, 13, 15, 0, 6, 8, 0, 14, 16
        ], [8, 1, 3, 1]);
        const blockShape = [2, 2];
        const crops = [[0, 0], [2, 0]];
        const res = tf.batchToSpaceND(t, blockShape, crops);
        expect(res.shape).toEqual([2, 2, 4, 1]);
        expectArraysClose(await res.data(), [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16]);
    });
    it('tensor2d, blockShape [1]', async () => {
        const t = tf.tensor2d([1, 2, 3, 4], [2, 2]);
        const blockShape = [2];
        const crops = [[0, 0]];
        const res = tf.batchToSpaceND(t, blockShape, crops);
        expect(res.shape).toEqual([1, 4]);
        expectArraysClose(await res.data(), [1, 3, 2, 4]);
    });
    it('tensor3d,  blockSHape [1]', async () => {
        const t = tf.tensor([
            -61, 37, -68, 72, 31, 62, 0, -13, 28, 54, 96,
            44, -55, -64, -88, -94, 65, -32, -96, -73, -2, -77,
            -14, 47, 33, 15, 70, 20, 75, 28, 84, -13
        ], [8, 2, 2]);
        const blockShape = [2];
        const crops = [[0, 2]];
        const res = tf.batchToSpaceND(t, blockShape, crops);
        expect(res.shape).toEqual([4, 2, 2]);
        expectArraysClose(await res.data(), [-61, 37, 65, -32, 31, 62, -2, -77, 28, 54, 33, 15, -55, -64, 75, 28]);
    });
    it('tensor3d, blockShape [2]', async () => {
        const t = tf.tensor([
            -61, 37, -68, 72, 31, 62, 0, -13, 28, 54, 96,
            44, -55, -64, -88, -94, 65, -32, -96, -73, -2, -77,
            -14, 47, 33, 15, 70, 20, 75, 28, 84, -13
        ], [8, 2, 2]);
        const blockShape = [2, 2];
        const crops = [[2, 0], [2, 0]];
        const res = tf.batchToSpaceND(t, blockShape, crops);
        expect(res.shape).toEqual([2, 2, 2]);
        expectArraysClose(await res.data(), [72, 44, -73, 20, -13, -94, 47, -13]);
    });
    it('throws when blockShape equal to input rank', () => {
        const t = tf.tensor4d([1, 2, 3, 4], [4, 1, 1, 1]);
        const blockShape = [2, 2, 2, 2];
        const crops = [[0, 0], [0, 0], [0, 0], [0, 0]];
        expect(() => tf.batchToSpaceND(t, blockShape, crops))
            .toThrowError(`input rank is ${t.rank} but should be > than blockShape.length ${blockShape.length}`);
    });
    it('throws when crops row dimension not equal to blockshape', () => {
        const t = tf.tensor4d([1, 2, 3, 4], [4, 1, 1, 1]);
        const blockShape = [2, 2];
        const crops = [[0, 0]];
        expect(() => tf.batchToSpaceND(t, blockShape, crops))
            .toThrowError(`crops.length is ${crops.length} but should be equal to blockShape.length  ${blockShape.length}`);
    });
    it('throws when input tensor batch not divisible by prod(blockShape)', () => {
        const t = tf.tensor4d([1, 2, 3, 4, 5], [5, 1, 1, 1]);
        const blockShape = [2, 2];
        const crops = [[0, 0], [0, 0]];
        const prod = blockShape.reduce((a, b) => a * b);
        expect(() => tf.batchToSpaceND(t, blockShape, crops))
            .toThrowError(`input tensor batch is ${t.shape[0]} but is not divisible by the ` +
            `product of the elements of blockShape ${blockShape.join(' * ')} === ${prod}`);
    });
    it('accepts a tensor-like object', async () => {
        const t = [[[[1]]], [[[2]]], [[[3]]], [[[4]]]];
        const blockShape = [2, 2];
        const crops = [[0, 0], [0, 0]];
        const res = tf.batchToSpaceND(t, blockShape, crops);
        expect(res.shape).toEqual([1, 2, 2, 1]);
        expectArraysClose(await res.data(), [1, 2, 3, 4]);
    });
    it('gradients,  input shape=[4, 2, 2], block shape=[2]', async () => {
        const t = tf.tensor([-61, 37, -68, 72, 31, 62, 0, -13, 28, 54, 96, 44, -55, -64, -88, -94], [4, 2, 2]);
        const blockShape = [2];
        const crops = [[0, 2]];
        const dy = tf.tensor([.01, .02, .03, .04, .05, .06, .07, .08], [2, 2, 2]);
        const gradient = tf.grad(t => tf.batchToSpaceND(t, blockShape, crops))(t, dy);
        expect(gradient.shape).toEqual([4, 2, 2]);
        expectArraysClose(await gradient.data(), [
            0.01, 0.02, 0, 0, 0.05, 0.06, 0, 0, 0.03, 0.04, 0, 0, 0.07, 0.08, 0, 0
        ]);
    });
    it('gradients, input shape=[4, 2, 2, 1], block shape=[2, 2]', async () => {
        const t = tf.tensor4d([1, 3, 9, 11, 2, 4, 10, 12, 5, 7, 13, 15, 6, 8, 14, 16], [4, 2, 2, 1]);
        const blockShape = [2, 2];
        const crops = [[0, 0], [0, 0]];
        const dy = tf.tensor([1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16], [1, 4, 4, 1]);
        const gradient = tf.grad(t => tf.batchToSpaceND(t, blockShape, crops))(t, dy);
        expect(gradient.shape).toEqual([4, 2, 2, 1]);
        expectArraysClose(await gradient.data(), [1, 3, 9, 11, 2, 4, 10, 12, 5, 7, 13, 15, 6, 8, 14, 16]);
    });
    it('gradient with clones, input=[4, 2, 2, 1], block shape=[2, 2]', async () => {
        const t = tf.tensor4d([1, 3, 9, 11, 2, 4, 10, 12, 5, 7, 13, 15, 6, 8, 14, 16], [4, 2, 2, 1]);
        const blockShape = [2, 2];
        const crops = [[0, 0], [0, 0]];
        const dy = tf.tensor([1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16], [1, 4, 4, 1]);
        const gradient = tf.grad(t => tf.batchToSpaceND(t.clone(), blockShape, crops).clone())(t, dy);
        expect(gradient.shape).toEqual([4, 2, 2, 1]);
        expectArraysClose(await gradient.data(), [1, 3, 9, 11, 2, 4, 10, 12, 5, 7, 13, 15, 6, 8, 14, 16]);
    });
});
//# sourceMappingURL=data:application/json;base64,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