/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { BROWSER_ENVS, describeWithFlags } from '../jasmine_util';
import { BrowserIndexedDB, browserIndexedDB } from './indexed_db';
import { BrowserLocalStorage, browserLocalStorage } from './local_storage';
import { IORouterRegistry } from './router_registry';
describeWithFlags('IORouterRegistry', BROWSER_ENVS, () => {
    const localStorageRouter = (url) => {
        const scheme = 'localstorage://';
        if (url.startsWith(scheme)) {
            return browserLocalStorage(url.slice(scheme.length));
        }
        else {
            return null;
        }
    };
    const indexedDBRouter = (url) => {
        const scheme = 'indexeddb://';
        if (url.startsWith(scheme)) {
            return browserIndexedDB(url.slice(scheme.length));
        }
        else {
            return null;
        }
    };
    class FakeIOHandler {
        constructor(url1, url2) { }
    }
    const fakeMultiStringRouter = (url) => {
        const scheme = 'foo://';
        if (Array.isArray(url) && url.length === 2) {
            if (url[0].startsWith(scheme) && url[1].startsWith(scheme)) {
                return new FakeIOHandler(url[0], url[1]);
            }
            else {
                return null;
            }
        }
        else {
            return null;
        }
    };
    let tempRegistryInstance = null;
    beforeEach(() => {
        // Force reset registry for testing.
        // tslint:disable:no-any
        tempRegistryInstance = IORouterRegistry.instance;
        IORouterRegistry.instance = null;
        // tslint:enable:no-any
    });
    afterEach(() => {
        // tslint:disable-next-line:no-any
        IORouterRegistry.instance = tempRegistryInstance;
    });
    it('getSaveHandler succeeds', () => {
        IORouterRegistry.registerSaveRouter(localStorageRouter);
        IORouterRegistry.registerSaveRouter(indexedDBRouter);
        const out1 = tf.io.getSaveHandlers('localstorage://foo-model');
        expect(out1.length).toEqual(1);
        expect(out1[0] instanceof BrowserLocalStorage).toEqual(true);
        const out2 = tf.io.getSaveHandlers('indexeddb://foo-model');
        expect(out2.length).toEqual(1);
        expect(out2[0] instanceof BrowserIndexedDB).toEqual(true);
    });
    it('getLoadHandler succeeds', () => {
        IORouterRegistry.registerLoadRouter(localStorageRouter);
        IORouterRegistry.registerLoadRouter(indexedDBRouter);
        const out1 = tf.io.getLoadHandlers('localstorage://foo-model');
        expect(out1.length).toEqual(1);
        expect(out1[0] instanceof BrowserLocalStorage).toEqual(true);
        const out2 = tf.io.getLoadHandlers('indexeddb://foo-model');
        expect(out2.length).toEqual(1);
        expect(out2[0] instanceof BrowserIndexedDB).toEqual(true);
    });
    it('getLoadHandler with string array argument succeeds', () => {
        IORouterRegistry.registerLoadRouter(fakeMultiStringRouter);
        const loadHandler = IORouterRegistry.getLoadHandlers(['foo:///123', 'foo:///456']);
        expect(loadHandler[0] instanceof FakeIOHandler).toEqual(true);
        expect(IORouterRegistry.getLoadHandlers([
            'foo:///123', 'bar:///456'
        ])).toEqual([]);
        expect(IORouterRegistry.getLoadHandlers(['foo:///123'])).toEqual([]);
        expect(IORouterRegistry.getLoadHandlers('foo:///123')).toEqual([]);
    });
    it('getSaveHandler fails', () => {
        IORouterRegistry.registerSaveRouter(localStorageRouter);
        expect(tf.io.getSaveHandlers('invalidscheme://foo-model')).toEqual([]);
        // Check there is no crosstalk between save and load handlers.
        expect(tf.io.getLoadHandlers('localstorage://foo-model')).toEqual([]);
    });
    const fakeLoadOptionsRouter = (url, loadOptions) => {
        return new FakeLoadOptionsHandler(url, loadOptions);
    };
    class FakeLoadOptionsHandler {
        constructor(url, loadOptions) {
            this.loadOptions = loadOptions;
        }
        get loadOptionsData() {
            return this.loadOptions;
        }
    }
    it('getLoadHandler loadOptions', () => {
        IORouterRegistry.registerLoadRouter(fakeLoadOptionsRouter);
        const loadOptions = {
            onProgress: (fraction) => { },
            fetchFunc: () => { }
        };
        const loadHandler = tf.io.getLoadHandlers('foo:///123', loadOptions);
        expect(loadHandler.length).toEqual(1);
        expect(loadHandler[0] instanceof FakeLoadOptionsHandler).toEqual(true);
        // Check callback function passed to IOHandler
        expect(loadHandler[0].loadOptionsData)
            .toBe(loadOptions);
    });
});
//# sourceMappingURL=data:application/json;base64,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