/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
export const EPSILON_FLOAT32 = 1e-7;
export const EPSILON_FLOAT16 = 1e-4;
/** Convenient class for storing tensor-related data. */
export class DataStorage {
    constructor(backend, dataMover) {
        this.backend = backend;
        this.dataMover = dataMover;
        this.data = new WeakMap();
        this.dataIdsCount = 0;
    }
    get(dataId) {
        if (!this.data.has(dataId)) {
            this.dataMover.moveData(this.backend, dataId);
        }
        return this.data.get(dataId);
    }
    set(dataId, value) {
        this.dataIdsCount++;
        this.data.set(dataId, value);
    }
    has(dataId) {
        return this.data.has(dataId);
    }
    delete(dataId) {
        this.dataIdsCount--;
        return this.data.delete(dataId);
    }
    numDataIds() {
        return this.dataIdsCount;
    }
}
/**
 * The interface that defines the kernels that should be implemented when
 * adding a new backend. New backends don't need to implement every one of the
 * methods, this can be done gradually (throw an error for unimplemented
 * methods).
 */
export class KernelBackend {
    refCount(dataId) {
        return notYetImplemented('refCount');
    }
    incRef(dataId) {
        return notYetImplemented('incRef');
    }
    timerAvailable() {
        return true;
    }
    time(f) {
        return notYetImplemented('time');
    }
    read(dataId) {
        return notYetImplemented('read');
    }
    readSync(dataId) {
        return notYetImplemented('readSync');
    }
    readToGPU(dataId, options) {
        return notYetImplemented('readToGPU');
    }
    numDataIds() {
        return notYetImplemented('numDataIds');
    }
    disposeData(dataId, force) {
        return notYetImplemented('disposeData');
    }
    write(values, shape, dtype) {
        return notYetImplemented('write');
    }
    move(dataId, values, shape, dtype, refCount) {
        return notYetImplemented('move');
    }
    memory() {
        return notYetImplemented('memory');
    }
    /** Returns the highest precision for floats in bits (e.g. 16 or 32) */
    floatPrecision() {
        return notYetImplemented('floatPrecision');
    }
    /** Returns the smallest representable number.  */
    epsilon() {
        return this.floatPrecision() === 32 ? EPSILON_FLOAT32 : EPSILON_FLOAT16;
    }
    dispose() {
        return notYetImplemented('dispose');
    }
}
function notYetImplemented(kernelName) {
    throw new Error(`'${kernelName}' not yet implemented or not found in the registry. ` +
        `This kernel may not be supported by the tfjs backend you have chosen`);
}
//# sourceMappingURL=data:application/json;base64,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