/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { scatter_util, ScatterNd, util } from '@tensorflow/tfjs-core';
import { CppDType } from './types';
let wasmScatterNd;
function setup(backend) {
    wasmScatterNd = backend.wasm.cwrap(ScatterNd, null /*void*/, [
        'number',
        'number',
        'number',
        'number',
        'number',
        'number',
        'array',
        'number',
        'number' // outId
    ]);
}
function scatterNd(args) {
    const { backend, inputs, attrs } = args;
    const { indices, updates } = inputs;
    const { shape } = attrs;
    const out = backend.makeOutput(shape, updates.dtype);
    if (util.sizeFromShape(shape) === 0) {
        return out;
    }
    const { sliceRank, numUpdates, sliceSize, strides, outputSize } = scatter_util.calculateShapes(updates, indices, shape);
    const indicesData = backend.dataIdMap.get(indices.dataId);
    const indicesId = indicesData.id;
    const updatesData = backend.dataIdMap.get(updates.dataId);
    const updatesId = updatesData.id;
    const stridesBytes = new Uint8Array(new Int32Array(strides).buffer);
    const outId = backend.dataIdMap.get(out.dataId).id;
    wasmScatterNd(indicesId, updatesId, CppDType[updates.dtype], sliceRank, numUpdates, sliceSize, stridesBytes, outputSize, outId);
    return out;
}
export const scatterNdConfig = {
    kernelName: ScatterNd,
    backendName: 'wasm',
    setupFunc: setup,
    kernelFunc: scatterNd
};
//# sourceMappingURL=ScatterNd.js.map