"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkRequestEvent = exports.ResponseWrapperXhr = exports.ResponseWrapperFetch = exports.RequestWrapperXhr = exports.RequestWrapperFetch = exports.MAXIMUM_ENTRIES = void 0;
var tslib_1 = require("tslib");
var global_scope_1 = require("./global-scope");
exports.MAXIMUM_ENTRIES = 100;
/**
 * This class encapsulates the RequestInit (https://developer.mozilla.org/en-US/docs/Web/API/RequestInit)
 * object so that the consumer can only get access to the headers, method and body size.
 *
 * This is to prevent consumers from directly accessing the Request object
 * and mutating it or running costly operations on it.
 *
 * IMPORTANT:
 *    * Do not make changes to this class without careful consideration
 *      of performance implications, memory usage and potential to mutate the customer's
 *      request.
 *   * NEVER .clone() the RequestInit object. This will 2x's the memory overhead of the request
 *   * NEVER: call .arrayBuffer(), text(), json() or any other method on the body that
 *     consumes the body's stream. This will cause the response to be consumed
 *     meaning the body will be empty when the customer tries to access it.
 *     (ie: if the body is an instanceof https://developer.mozilla.org/en-US/docs/Web/API/ReadableStream
 *      never call any of the methods on it)
 */
var RequestWrapperFetch = /** @class */ (function () {
    function RequestWrapperFetch(request) {
        this.request = request;
    }
    Object.defineProperty(RequestWrapperFetch.prototype, "headers", {
        get: function () {
            if (this._headers)
                return this._headers;
            var headersUnsafe = this.request.headers;
            if (Array.isArray(headersUnsafe)) {
                var headers = headersUnsafe;
                this._headers = headers.reduce(function (acc, _a) {
                    var _b = tslib_1.__read(_a, 2), key = _b[0], value = _b[1];
                    acc[key] = value;
                    return acc;
                }, {});
            }
            else if (headersUnsafe instanceof Headers) {
                var headersSafe = headersUnsafe;
                var headersObj_1 = {};
                headersSafe.forEach(function (value, key) {
                    headersObj_1[key] = value;
                });
                this._headers = headersObj_1;
            }
            else if (typeof headersUnsafe === 'object') {
                this._headers = headersUnsafe;
            }
            return this._headers;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(RequestWrapperFetch.prototype, "bodySize", {
        get: function () {
            if (typeof this._bodySize === 'number')
                return this._bodySize;
            var global = (0, global_scope_1.getGlobalScope)();
            /* istanbul ignore if */
            if (!(global === null || global === void 0 ? void 0 : global.TextEncoder)) {
                return;
            }
            var body = this.request.body;
            this._bodySize = getBodySize(body, exports.MAXIMUM_ENTRIES);
            return this._bodySize;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(RequestWrapperFetch.prototype, "method", {
        get: function () {
            return this.request.method;
        },
        enumerable: false,
        configurable: true
    });
    return RequestWrapperFetch;
}());
exports.RequestWrapperFetch = RequestWrapperFetch;
var RequestWrapperXhr = /** @class */ (function () {
    function RequestWrapperXhr(body) {
        this.body = body;
    }
    Object.defineProperty(RequestWrapperXhr.prototype, "bodySize", {
        get: function () {
            return getBodySize(this.body, exports.MAXIMUM_ENTRIES);
        },
        enumerable: false,
        configurable: true
    });
    return RequestWrapperXhr;
}());
exports.RequestWrapperXhr = RequestWrapperXhr;
function getBodySize(bodyUnsafe, maxEntries) {
    var e_1, _a;
    var bodySize;
    var global = (0, global_scope_1.getGlobalScope)();
    /* istanbul ignore next */
    var TextEncoder = global === null || global === void 0 ? void 0 : global.TextEncoder;
    /* istanbul ignore next */
    if (!TextEncoder) {
        return;
    }
    var bodySafe;
    if (typeof bodyUnsafe === 'string') {
        bodySafe = bodyUnsafe;
        bodySize = new TextEncoder().encode(bodySafe).length;
    }
    else if (bodyUnsafe instanceof Blob) {
        bodySafe = bodyUnsafe;
        bodySize = bodySafe.size;
    }
    else if (bodyUnsafe instanceof URLSearchParams) {
        bodySafe = bodyUnsafe;
        bodySize = new TextEncoder().encode(bodySafe.toString()).length;
    }
    else if (ArrayBuffer.isView(bodyUnsafe)) {
        bodySafe = bodyUnsafe;
        bodySize = bodySafe.byteLength;
    }
    else if (bodyUnsafe instanceof ArrayBuffer) {
        bodySafe = bodyUnsafe;
        bodySize = bodySafe.byteLength;
    }
    else if (bodyUnsafe instanceof FormData) {
        // Estimating only for text parts; not accurate for files
        var formData = bodyUnsafe;
        var total = 0;
        var count = 0;
        try {
            for (var _b = tslib_1.__values(formData.entries()), _c = _b.next(); !_c.done; _c = _b.next()) {
                var _d = tslib_1.__read(_c.value, 2), key = _d[0], value = _d[1];
                total += key.length;
                if (typeof value === 'string') {
                    total += new TextEncoder().encode(value).length;
                }
                else if (value instanceof Blob) {
                    total += value.size;
                }
                else {
                    // encountered an unknown type
                    // we can't estimate the size of this entry
                    return;
                }
                // terminate if we reach the maximum number of entries
                // to avoid performance issues in case of very large FormData
                if (++count >= maxEntries) {
                    return;
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
            }
            finally { if (e_1) throw e_1.error; }
        }
        bodySize = total;
    }
    else if (bodyUnsafe instanceof ReadableStream) {
        // If bodyUnsafe is an instanceof ReadableStream, we can't determine the size,
        // without consuming it, so we return undefined.
        // Never ever consume ReadableStream! DO NOT DO IT!!!
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        bodySafe = bodyUnsafe;
        return;
    }
    return bodySize;
}
/**
 * This class encapsulates the Fetch API Response object
 * (https://developer.mozilla.org/en-US/docs/Web/API/Response) so that the consumer can
 * only get access to the headers and body size.
 *
 * This is to prevent consumers from directly accessing the Response object
 * and mutating it or running costly operations on it.
 *
 * IMPORTANT:
 *   * Do not make changes to this class without careful consideration
 *     of performance implications, memory usage and potential to mutate the customer's
 *     response.
 *   * NEVER .clone() the Response object. This will 2x's the memory overhead of the response
 *   * NEVER consume the body's stream. This will cause the response to be consumed
 *     meaning the body will be empty when the customer tries to access it.
 *     (ie: if the body is an instanceof https://developer.mozilla.org/en-US/docs/Web/API/ReadableStream
 *      never call any of the methods on it)
 */
var ResponseWrapperFetch = /** @class */ (function () {
    function ResponseWrapperFetch(response) {
        this.response = response;
    }
    Object.defineProperty(ResponseWrapperFetch.prototype, "headers", {
        get: function () {
            var _a;
            if (this._headers)
                return this._headers;
            if (this.response.headers instanceof Headers) {
                var headersSafe = this.response.headers;
                var headersOut_1 = {};
                /* istanbul ignore next */
                (_a = headersSafe === null || headersSafe === void 0 ? void 0 : headersSafe.forEach) === null || _a === void 0 ? void 0 : _a.call(headersSafe, function (value, key) {
                    headersOut_1[key] = value;
                });
                this._headers = headersOut_1;
                return headersOut_1;
            }
            return;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ResponseWrapperFetch.prototype, "bodySize", {
        get: function () {
            var _a, _b;
            if (this._bodySize !== undefined)
                return this._bodySize;
            /* istanbul ignore next */
            var contentLength = (_b = (_a = this.response.headers) === null || _a === void 0 ? void 0 : _a.get) === null || _b === void 0 ? void 0 : _b.call(_a, 'content-length');
            var bodySize = contentLength ? parseInt(contentLength, 10) : undefined;
            this._bodySize = bodySize;
            return bodySize;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ResponseWrapperFetch.prototype, "status", {
        get: function () {
            return this.response.status;
        },
        enumerable: false,
        configurable: true
    });
    return ResponseWrapperFetch;
}());
exports.ResponseWrapperFetch = ResponseWrapperFetch;
var ResponseWrapperXhr = /** @class */ (function () {
    function ResponseWrapperXhr(statusCode, headersString, size) {
        this.statusCode = statusCode;
        this.headersString = headersString;
        this.size = size;
    }
    Object.defineProperty(ResponseWrapperXhr.prototype, "bodySize", {
        get: function () {
            return this.size;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ResponseWrapperXhr.prototype, "status", {
        get: function () {
            return this.statusCode;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ResponseWrapperXhr.prototype, "headers", {
        get: function () {
            var e_2, _a;
            if (!this.headersString) {
                return;
            }
            var headers = {};
            var headerLines = this.headersString.split('\r\n');
            try {
                for (var headerLines_1 = tslib_1.__values(headerLines), headerLines_1_1 = headerLines_1.next(); !headerLines_1_1.done; headerLines_1_1 = headerLines_1.next()) {
                    var line = headerLines_1_1.value;
                    var _b = tslib_1.__read(line.split(': '), 2), key = _b[0], value = _b[1];
                    if (key && value) {
                        headers[key] = value;
                    }
                }
            }
            catch (e_2_1) { e_2 = { error: e_2_1 }; }
            finally {
                try {
                    if (headerLines_1_1 && !headerLines_1_1.done && (_a = headerLines_1.return)) _a.call(headerLines_1);
                }
                finally { if (e_2) throw e_2.error; }
            }
            return headers;
        },
        enumerable: false,
        configurable: true
    });
    return ResponseWrapperXhr;
}());
exports.ResponseWrapperXhr = ResponseWrapperXhr;
var NetworkRequestEvent = /** @class */ (function () {
    function NetworkRequestEvent(type, method, timestamp, startTime, url, requestWrapper, status, duration, responseWrapper, error, endTime) {
        if (status === void 0) { status = 0; }
        this.type = type;
        this.method = method;
        this.timestamp = timestamp;
        this.startTime = startTime;
        this.url = url;
        this.requestWrapper = requestWrapper;
        this.status = status;
        this.duration = duration;
        this.responseWrapper = responseWrapper;
        this.error = error;
        this.endTime = endTime;
    }
    NetworkRequestEvent.prototype.toSerializable = function () {
        var _a, _b, _c, _d;
        var serialized = {
            type: this.type,
            method: this.method,
            url: this.url,
            timestamp: this.timestamp,
            status: this.status,
            duration: this.duration,
            error: this.error,
            startTime: this.startTime,
            endTime: this.endTime,
            requestHeaders: (_a = this.requestWrapper) === null || _a === void 0 ? void 0 : _a.headers,
            requestBodySize: (_b = this.requestWrapper) === null || _b === void 0 ? void 0 : _b.bodySize,
            responseHeaders: (_c = this.responseWrapper) === null || _c === void 0 ? void 0 : _c.headers,
            responseBodySize: (_d = this.responseWrapper) === null || _d === void 0 ? void 0 : _d.bodySize,
        };
        return Object.fromEntries(Object.entries(serialized).filter(function (_a) {
            var _b = tslib_1.__read(_a, 2), _ = _b[0], v = _b[1];
            return v !== undefined;
        }));
    };
    return NetworkRequestEvent;
}());
exports.NetworkRequestEvent = NetworkRequestEvent;
//# sourceMappingURL=network-request-event.js.map