import { Plugin } from './types/plugin';
import { IConfig } from './config';
import { BaseEvent, EventOptions } from './types/event/base-event';
import { Result } from './types/result';
import { Event, UserProperties } from './types/event/event';
import { IIdentify } from './identify';
import { IRevenue } from './revenue';
import { Timeline } from './timeline';
import { AmplitudeReturn, returnWrapper } from './utils/return-wrapper';
interface PluginHost {
    plugin(name: string): Plugin | undefined;
    plugins<T extends Plugin>(pluginClass: new (...args: any[]) => T): T[];
}
export interface CoreClient {
    /**
     * Adds a new plugin.
     *
     * ```typescript
     * const plugin = {
     *   name: 'myPlugin',
     *   type: 'enrichment',
     *   setup(config: Config) {
     *     return;
     *   },
     *   execute(context: Event) {
     *     return context;
     *   },
     * };
     * amplitude.add(plugin);
     * ```
     */
    add(plugin: Plugin): AmplitudeReturn<void>;
    /**
     * Removes a plugin.
     *
     * ```typescript
     * amplitude.remove('myPlugin');
     * ```
     */
    remove(pluginName: string): AmplitudeReturn<void>;
    /**
     * Tracks user-defined event, with specified type, optional event properties and optional overwrites.
     *
     * ```typescript
     * // event tracking with event type only
     * track('Page Load');
     *
     * // event tracking with event type and additional event properties
     * track('Page Load', { loadTime: 1000 });
     *
     * // event tracking with event type, additional event properties, and overwritten event options
     * track('Page Load', { loadTime: 1000 }, { sessionId: -1 });
     *
     * // alternatively, this tracking method is awaitable
     * const result = await track('Page Load').promise;
     * console.log(result.event); // {...}
     * console.log(result.code); // 200
     * console.log(result.message); // "Event tracked successfully"
     * ```
     */
    track(eventInput: BaseEvent | string, eventProperties?: Record<string, any>, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    /**
     * Alias for track()
     */
    logEvent(eventInput: BaseEvent | string, eventProperties?: Record<string, any>, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    /**
     * Sends an identify event containing user property operations
     *
     * ```typescript
     * const id = new Identify();
     * id.set('colors', ['rose', 'gold']);
     * identify(id);
     *
     * // alternatively, this tracking method is awaitable
     * const result = await identify(id).promise;
     * console.log(result.event); // {...}
     * console.log(result.code); // 200
     * console.log(result.message); // "Event tracked successfully"
     * ```
     */
    identify(identify: IIdentify, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    /**
     * Sends a group identify event containing group property operations.
     *
     * ```typescript
     * const id = new Identify();
     * id.set('skills', ['js', 'ts']);
     * const groupType = 'org';
     * const groupName = 'engineering';
     * groupIdentify(groupType, groupName, id);
     *
     * // alternatively, this tracking method is awaitable
     * const result = await groupIdentify(groupType, groupName, id).promise;
     * console.log(result.event); // {...}
     * console.log(result.code); // 200
     * console.log(result.message); // "Event tracked successfully"
     * ```
     */
    groupIdentify(groupType: string, groupName: string | string[], identify: IIdentify, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    /**
     * Assigns a user to group
     *
     * ```typescript
     * const groupType = 'orgId';
     * const groupName = '15';
     * setGroup(groupType, groupName, { user_id: '12345' })
     * ```
     */
    setGroup(groupType: string, groupName: string | string[], eventOptions?: EventOptions): AmplitudeReturn<Result>;
    /**
     * Sends a revenue event containing revenue property operations.
     *
     * ```typescript
     * const rev = new Revenue();
     * rev.setRevenue(100);
     * revenue(rev);
     *
     * // alternatively, this tracking method is awaitable
     * const result = await revenue(rev).promise;
     * console.log(result.event); // {...}
     * console.log(result.code); // 200
     * console.log(result.message); // "Event tracked successfully"
     * ```
     */
    revenue(revenue: IRevenue, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    /**
     * Sets a new optOut config value. This toggles event tracking on/off.
     *
     *```typescript
     * // Stops tracking
     * setOptOut(true);
     *
     * // Starts/resumes tracking
     * setOptOut(false);
     * ```
     */
    setOptOut(optOut: boolean): void;
    /**
     * Flush all unsent events.
     *
     *```typescript
     * flush();
     * ```
     */
    flush(): AmplitudeReturn<void>;
}
export declare class AmplitudeCore implements CoreClient, PluginHost {
    protected initializing: boolean;
    protected name: string;
    config: IConfig;
    timeline: Timeline;
    isReady: boolean;
    protected q: Array<CallableFunction | typeof returnWrapper>;
    protected dispatchQ: Array<CallableFunction>;
    constructor(name?: string);
    protected _init(config: IConfig): Promise<void>;
    runQueuedFunctions(queueName: 'q' | 'dispatchQ'): Promise<void>;
    track(eventInput: BaseEvent | string, eventProperties?: Record<string, any>, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    logEvent: (eventInput: BaseEvent | string, eventProperties?: Record<string, any>, eventOptions?: EventOptions) => AmplitudeReturn<Result>;
    identify(identify: IIdentify, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    groupIdentify(groupType: string, groupName: string | string[], identify: IIdentify, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    setGroup(groupType: string, groupName: string | string[], eventOptions?: EventOptions): AmplitudeReturn<Result>;
    revenue(revenue: IRevenue, eventOptions?: EventOptions): AmplitudeReturn<Result>;
    add(plugin: Plugin): AmplitudeReturn<void>;
    _addPlugin(plugin: Plugin): AmplitudeReturn<void>;
    remove(pluginName: string): AmplitudeReturn<void>;
    _removePlugin(pluginName: string): AmplitudeReturn<void>;
    dispatchWithCallback(event: Event, callback: (result: Result) => void): void;
    dispatch(event: Event): Promise<Result>;
    /**
     *
     * This method applies identify operations to user properties and
     * returns a single object representing the final user property state.
     *
     * This is a best-effort api that only supports $set, $clearAll, and $unset.
     * Other operations are not supported and are ignored.
     *
     *
     * @param userProperties The `event.userProperties` object from an Identify event.
     * @returns A key-value object user properties without operations.
     *
     * @example
     * Input:
     * {
     *   $set: { plan: 'premium' },
     *   custom_flag: true
     * }
     *
     * Output:
     * {
     *   plan: 'premium',
     *   custom_flag: true
     * }
     */
    getOperationAppliedUserProperties(userProperties: UserProperties | undefined): {
        [key: string]: any;
    };
    process(event: Event): Promise<Result>;
    setOptOut(optOut: boolean): void;
    _setOptOut(optOut: boolean): void;
    flush(): AmplitudeReturn<void>;
    plugin(name: string): Plugin | undefined;
    plugins<T extends Plugin>(pluginClass: {
        new (...args: any[]): T;
    }): T[];
}
export {};
//# sourceMappingURL=core-client.d.ts.map