/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from './index';
import { ALL_ENVS, describeWithFlags, SYNC_BACKEND_ENVS } from './jasmine_util';
import { checkComputationForErrors, Logger, Profiler } from './profiler';
class TestBackendTimer {
    constructor(delayMs, queryTimeMs, extraInfo) {
        this.delayMs = delayMs;
        this.queryTimeMs = queryTimeMs;
        this.extraInfo = extraInfo;
        this.counter = 1;
    }
    timerAvailable() {
        return true;
    }
    async time(query) {
        query();
        const kernelMs = await new Promise(resolve => setTimeout(() => resolve(this.queryTimeMs * this.counter++), this.delayMs));
        return { kernelMs, getExtraProfileInfo: () => this.extraInfo };
    }
}
class TestLogger extends Logger {
    logKernelProfile(name, result, vals, timeMs) { }
}
function promiseCheckWrapper(acturalValPromise, truthVal) {
    return acturalValPromise.then(acturalVal => {
        expect(acturalVal).toEqual(truthVal);
    });
}
function checkKernelProfile(acturalVal, truthVal) {
    expect(acturalVal.kernelName).toBe(truthVal.kernelName);
    expect(acturalVal.inputs).toBe(truthVal.inputs);
    acturalVal.outputs.forEach((output, index) => {
        expect(output).toBe(truthVal.outputs[index]);
    });
    const promiseContainer = [
        promiseCheckWrapper(acturalVal.timeMs, truthVal.timeMs),
        promiseCheckWrapper(acturalVal.extraInfo, truthVal.extraInfo),
    ];
    return Promise.all(promiseContainer);
}
describeWithFlags('profiler.Profiler', SYNC_BACKEND_ENVS, () => {
    it('profiles simple function', doneFn => {
        const delayMs = 5;
        const queryTimeMs = 10;
        const inputs = { 'x': tf.tensor1d([1]) };
        const extraInfo = '';
        const timer = new TestBackendTimer(delayMs, queryTimeMs, extraInfo);
        const logger = new TestLogger();
        const profiler = new Profiler(timer, logger);
        spyOn(timer, 'time').and.callThrough();
        spyOn(logger, 'logKernelProfile').and.callThrough();
        const timeSpy = timer.time;
        let kernelCalled = false;
        const result = 1;
        const resultScalar = tf.scalar(result);
        const kernelProfile = profiler.profileKernel('MatMul', inputs, () => {
            kernelCalled = true;
            return [resultScalar];
        });
        setTimeout(() => {
            expect(timeSpy.calls.count()).toBe(1);
            expect(kernelCalled).toBe(true);
            checkKernelProfile(kernelProfile, {
                kernelName: 'MatMul',
                outputs: [resultScalar],
                timeMs: queryTimeMs,
                inputs,
                extraInfo,
            }).then(() => doneFn());
        }, delayMs * 2);
    });
    it('profiles nested kernel with optional inputs', doneFn => {
        const delayMs = 5;
        const queryTimeMs = 10;
        const inputs = { 'x': tf.tensor1d([1]), 'bias': null };
        const extraInfo = '';
        const timer = new TestBackendTimer(delayMs, queryTimeMs, extraInfo);
        const logger = new TestLogger();
        const profiler = new Profiler(timer, logger);
        spyOn(timer, 'time').and.callThrough();
        spyOn(logger, 'logKernelProfile').and.callThrough();
        const timeSpy = timer.time;
        let matmulKernelCalled = false;
        let maxKernelCalled = false;
        const result = 1;
        const resultScalar = tf.scalar(result);
        let innerKernelProfile;
        const outerKernelProfile = profiler.profileKernel('MatMul', inputs, () => {
            innerKernelProfile = profiler.profileKernel('Max', inputs, () => {
                maxKernelCalled = true;
                return [resultScalar];
            });
            matmulKernelCalled = true;
            return innerKernelProfile.outputs;
        });
        setTimeout(() => {
            expect(timeSpy.calls.count()).toBe(2);
            expect(matmulKernelCalled).toBe(true);
            expect(maxKernelCalled).toBe(true);
            const checkInnerKernelProfile = checkKernelProfile(innerKernelProfile, {
                kernelName: 'Max',
                outputs: [resultScalar],
                timeMs: queryTimeMs,
                inputs,
                extraInfo
            });
            const checkOuterKernelProfile = checkKernelProfile(outerKernelProfile, {
                kernelName: 'MatMul',
                outputs: [resultScalar],
                timeMs: queryTimeMs * 2,
                inputs,
                extraInfo
            });
            Promise.all([checkInnerKernelProfile, checkOuterKernelProfile])
                .then(() => doneFn());
        }, delayMs * 2);
    });
    it('log kernelProfile', doneFn => {
        const delayMs = 5;
        const queryTimeMs = 10;
        const inputs = { 'x': tf.tensor1d([1]) };
        const extraInfo = '';
        const timer = new TestBackendTimer(delayMs, queryTimeMs, extraInfo);
        const logger = new TestLogger();
        const profiler = new Profiler(timer, logger);
        spyOn(logger, 'logKernelProfile').and.callThrough();
        const logKernelProfileSpy = logger.logKernelProfile;
        const result = 1;
        const resultScalar = tf.scalar(result);
        const kernelProfiles = profiler.profileKernel('MatMul', inputs, () => {
            return [resultScalar];
        });
        profiler.logKernelProfile(kernelProfiles);
        setTimeout(() => {
            expect(logKernelProfileSpy.calls.count()).toBe(1);
            expect(logKernelProfileSpy.calls.first().args).toEqual([
                'MatMul', resultScalar, new Float32Array([result]), queryTimeMs, inputs,
                extraInfo
            ]);
            doneFn();
        }, delayMs * 2);
    });
});
describe('profiler.checkComputationForErrors', () => {
    beforeAll(() => {
        // Silence warnings.
        spyOn(console, 'warn');
    });
    it('Float32Array has NaN', () => {
        expect(checkComputationForErrors(new Float32Array([1, 2, 3, NaN, 4, 255]), 'float32', 'test'))
            .toBe(true);
    });
    it('Float32Array has Infinity', () => {
        expect(checkComputationForErrors(new Float32Array([1, 2, 3, Infinity, 4, 255]), 'float32', 'test'))
            .toBe(true);
    });
    it('Float32Array no NaN', () => {
        // Int32 and Bool NaNs should not trigger an error.
        expect(checkComputationForErrors(new Float32Array([1, 2, 3, -1, 4, 255]), 'float32', 'test'))
            .toBe(false);
    });
});
describeWithFlags('profiler.Logger', ALL_ENVS, () => {
    it('skips logging for undefined input node in input tensor map', () => {
        const kernelName = 'FusedConv2D';
        const vals = new Float32Array(1);
        const outputs = tf.tensor1d([1]);
        const timeMs = 10;
        const inputs = {
            'x': tf.tensor1d([1]),
            'filter': tf.tensor1d([1]),
            'bias': tf.tensor1d([1]),
            'preluActivationWeights': undefined
        };
        const extraInfo = '';
        const logger = new Logger();
        spyOn(console, 'log');
        const consoleLogSpy = console.log;
        logger.logKernelProfile(kernelName, outputs, vals, timeMs, inputs, extraInfo);
        expect(consoleLogSpy.calls.first().args)
            .not.toContain('preluActivationWeights');
    });
});
//# sourceMappingURL=data:application/json;base64,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