/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../../index';
import { ALL_ENVS, describeWithFlags } from '../../jasmine_util';
import { expectArraysClose } from '../../test_util';
describeWithFlags('fused matmul', ALL_ENVS, () => {
    it('fused A x B', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const c = tf.fused.matMul({ a, b });
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0, 8, -3, 20]);
    });
    it('fused A x B with relu', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const transposeA = false;
        const transposeB = false;
        const c = tf.fused.matMul({ a, b, transposeA, transposeB, bias: null, activation: 'relu' });
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0, 8, 0, 20]);
    });
    it('fused A x B with elu', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const transposeA = false;
        const transposeB = false;
        const c = tf.fused.matMul({ a, b, transposeA, transposeB, bias: null, activation: 'elu' });
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0, 8, -0.9502, 20]);
    });
    it('fused A x B with relu6', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const transposeA = false;
        const transposeB = false;
        const c = tf.fused.matMul({ a, b, transposeA, transposeB, bias: null, activation: 'relu6' });
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0, 6, 0, 6]);
    });
    it('fused A x B with prelu', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const alpha = tf.tensor2d([0.5, 0.5], [1, 2]);
        const transposeA = false;
        const transposeB = false;
        const c = tf.fused.matMul({
            a,
            b,
            transposeA,
            transposeB,
            bias: null,
            activation: 'prelu',
            preluActivationWeights: alpha
        });
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0, 8, -1.5, 20]);
    });
    it('fused A x B with leakyrelu', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const alpha = 0.3;
        const transposeA = false;
        const transposeB = false;
        const c = tf.fused.matMul({
            a,
            b,
            transposeA,
            transposeB,
            bias: null,
            activation: 'leakyrelu',
            leakyreluAlpha: alpha
        });
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0, 8, -0.9000000357627869, 20]);
    });
    it('fused A x B with sigmoid', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const transposeA = false;
        const transposeB = false;
        const c = tf.fused.matMul({ a, b, transposeA, transposeB, bias: null, activation: 'sigmoid' });
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0.5, 0.99966455, 0.04742587, 1]);
    });
    it('fused A x B with relu transpose', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [2, 3]);
        const transposeA = false;
        const transposeB = true;
        const c = tf.fused.matMul({ a, b, transposeA, transposeB, bias: null, activation: 'relu' });
        expect(c.shape).toEqual([2, 2]);
        expectArraysClose(await c.data(), [0, 9, 0, 24]);
    });
    it('fused A x B with 2d bias and relu', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const c = tf.tensor2d([1, 1, 1, 1], [2, 2]);
        const transposeA = false;
        const transposeB = false;
        const d = tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: 'relu' });
        expect(d.shape).toEqual([2, 2]);
        expectArraysClose(await d.data(), [1, 9, 0, 21]);
    });
    it('fused A x B with relu and broadcasted bias', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const c = tf.tensor1d([1, 1]);
        const act = 'relu';
        const transposeA = false;
        const transposeB = false;
        const d = tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: act });
        expect(d.shape).toEqual([2, 2]);
        expectArraysClose(await d.data(), [1, 9, 0, 21]);
    });
    it('fused A x B with elu and broadcasted bias', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const c = tf.tensor1d([1, 1]);
        const act = 'elu';
        const transposeA = false;
        const transposeB = false;
        const d = tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: act });
        expect(d.shape).toEqual([2, 2]);
        expectArraysClose(await d.data(), [1, 9, -0.8647, 21]);
    });
    it('fused A x B with elu and broadcasted shape', async () => {
        const a = tf.tensor3d([1, 2, 3, 4, 5, 6], [1, 2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const c = tf.tensor1d([1, 1]);
        const act = 'elu';
        const transposeA = false;
        const transposeB = false;
        const d = tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: act });
        expect(d.shape).toEqual([1, 2, 2]);
        expectArraysClose(await d.data(), [1, 9, -0.8647, 21]);
    });
    it('fused A x B with relu and broadcasted bias different rank', async () => {
        const a = tf.tensor3d([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11], [2, 2, 3]);
        const b = tf.tensor3d([0, 1, -3, 2, 2, 1, 0, 1, -3, 2, 2, 1], [2, 3, 2]);
        const c = tf.tensor2d([1, 2], [1, 2]);
        const act = 'relu';
        const transposeA = false;
        const transposeB = false;
        const d = tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: act });
        expect(d.shape).toEqual([2, 2, 2]);
        expectArraysClose(await d.data(), [2, 6, 0, 18, 0, 30, 0, 42]);
    });
    it('fused A x B with 2d bias only', async () => {
        const a = tf.tensor2d([1, 2, 3, 4, 5, 6], [2, 3]);
        const b = tf.tensor2d([0, 1, -3, 2, 2, 1], [3, 2]);
        const c = tf.tensor2d([1, 1, 1, 1], [2, 2]);
        const transposeA = false;
        const transposeB = false;
        const d = tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: 'linear' });
        expect(d.shape).toEqual([2, 2]);
        expectArraysClose(await d.data(), [1, 9, -2, 21]);
    });
    it('fused A x B with relu gradient', async () => {
        const a = tf.tensor2d([1, 2, 3, 10, 20, -30], [2, 3]);
        const b = tf.tensor2d([2, 3, 4, -1, 2, 3], [3, 2]);
        const dy = tf.tensor2d([1, 10, 20, 30], [2, 2]);
        const transposeA = false;
        const transposeB = false;
        const grads = tf.grads((a, b) => {
            const prod = tf.matMul(a, b, transposeA, transposeB);
            return tf.relu(prod);
        });
        const fusedGrads = tf.grads((a, b) => {
            return tf.fused.matMul({ a, b, transposeA, transposeB, bias: null, activation: 'relu' });
        });
        const [da, db] = grads([a, b], dy);
        const [fusedDa, fusedDb] = fusedGrads([a, b], dy);
        expectArraysClose(await da.array(), await fusedDa.array());
        expectArraysClose(await db.data(), await fusedDb.array());
    });
    it('gradient with clones A x B with relu', () => {
        const a = tf.tensor2d([1, 2, 3, 10, 20, -30], [2, 3]);
        const b = tf.tensor2d([2, 3, 4, -1, 2, 3], [3, 2]);
        const dy = tf.tensor2d([1, 10, 20, 30], [2, 2]);
        const transposeA = false;
        const transposeB = false;
        const fusedGrads = tf.grads((a, b) => {
            return tf.fused
                .matMul({
                a: a.clone(),
                b: b.clone(),
                transposeA,
                transposeB,
                bias: null,
                activation: 'relu'
            })
                .clone();
        });
        const [fusedDa, fusedDb] = fusedGrads([a, b], dy);
        expect(fusedDa.shape).toEqual(a.shape);
        expect(fusedDb.shape).toEqual(b.shape);
    });
    it('fused A x B with relu bias gradient', async () => {
        const a = tf.tensor2d([1, 2, 3, 10, 20, -30], [2, 3]);
        const b = tf.tensor2d([2, 3, 4, -1, 2, 3], [3, 2]);
        const c = tf.tensor2d([1, 1, 1, 1], [2, 2]);
        const transposeA = false;
        const transposeB = false;
        const dy = tf.tensor2d([1, 10, 20, 30], [2, 2]);
        const grads = tf.grads((a, b, c) => {
            const prod = tf.matMul(a, b, transposeA, transposeB);
            const sum = tf.add(prod, c);
            return tf.relu(sum);
        });
        const fusedGrads = tf.grads((a, b, c) => {
            return tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: 'relu' });
        });
        const [da, db, dc] = grads([a, b, c], dy);
        const [fusedDa, fusedDb, fusedDc] = fusedGrads([a, b, c], dy);
        expectArraysClose(await da.array(), await fusedDa.array());
        expectArraysClose(await db.array(), await fusedDb.array());
        expectArraysClose(await dc.array(), await fusedDc.array());
    });
    it('fused A x B with relu bias gradient transpose', async () => {
        const a = tf.tensor2d([1, 2, 3, 10, 20, -30], [3, 2]);
        const b = tf.tensor2d([2, 3, 4, -1, 2, 3], [3, 2]);
        const c = tf.tensor2d([1, 1, 1, 1], [2, 2]);
        const transposeA = true;
        const transposeB = false;
        const dy = tf.tensor2d([1, 10, 20, 30], [2, 2]);
        const grads = tf.grads((a, b, c) => {
            const prod = tf.matMul(a, b, transposeA, transposeB);
            const sum = tf.add(prod, c);
            return tf.relu(sum);
        });
        const fusedGrads = tf.grads((a, b, c) => {
            return tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: 'relu' });
        });
        const [da, db, dc] = grads([a, b, c], dy);
        const [fusedDa, fusedDb, fusedDc] = fusedGrads([a, b, c], dy);
        expectArraysClose(await da.array(), await fusedDa.array());
        expectArraysClose(await db.array(), await fusedDb.array());
        expectArraysClose(await dc.array(), await fusedDc.array());
    });
    it('fused A x B with relu and broadcasted bias gradient', async () => {
        const a = tf.tensor2d([1, 2, 3, 10, 20, -30], [2, 3]);
        const b = tf.tensor2d([2, 3, 4, -1, 2, 3], [3, 2]);
        const c = tf.tensor2d([[1]]);
        const transposeA = false;
        const transposeB = false;
        const dy = tf.tensor2d([1, 10, 20, 30], [2, 2]);
        const grads = tf.grads((a, b, c) => {
            const prod = tf.matMul(a, b, transposeA, transposeB);
            const sum = tf.add(prod, c);
            return tf.relu(sum);
        });
        const fusedGrads = tf.grads((a, b, c) => {
            return tf.fused.matMul({ a, b, transposeA, transposeB, bias: c, activation: 'relu' });
        });
        const [da, db, dc] = grads([a, b, c], dy);
        const [fusedDa, fusedDb, fusedDc] = fusedGrads([a, b, c], dy);
        expectArraysClose(await da.array(), await fusedDa.array());
        expectArraysClose(await db.array(), await fusedDb.array());
        expectArraysClose(await dc.array(), await fusedDc.array());
    });
    it('fused matmul with relu6 and gradients', async () => {
        const a = tf.tensor2d([1, 2, 3, 10, 20, -30], [2, 3]);
        const b = tf.tensor2d([2, 3, 4, -1, 2, 3], [3, 2]);
        const dy = tf.tensor2d([1, 10, 20, 30], [2, 2]);
        const transposeA = false;
        const transposeB = false;
        const fusedGrads = tf.grads((a, b) => {
            return tf.fused.matMul({ a, b, transposeA, transposeB, bias: null, activation: 'relu6' });
        });
        const [fusedDa, fusedDb] = fusedGrads([a, b], dy);
        const grads = tf.grads((a, b) => {
            const prod = tf.matMul(a, b, transposeA, transposeB);
            return tf.relu6(prod);
        });
        const [da, db] = grads([a, b], dy);
        expectArraysClose(await da.array(), await fusedDa.array());
        expectArraysClose(await db.data(), await fusedDb.array());
    });
});
//# sourceMappingURL=data:application/json;base64,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