/**
 * @license
 * Copyright 2021 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
import { tensor1d, tensor2d, tensor3d } from './ops';
describeWithFlags('einsum', ALL_ENVS, () => {
    it('two scalars', async () => {
        const x = tf.scalar(2);
        const y = tf.scalar(3);
        const out = tf.einsum(',->', x, y);
        expectArraysClose(await out.data(), 6);
    });
    it('1D tensor and scalars: reduce', async () => {
        const x = tensor1d([2, 3]);
        const y = tf.scalar(4);
        const out = tf.einsum('i,->', x, y);
        expectArraysClose(await out.data(), 20);
    });
    it('1D tensor and scalars: multiply', async () => {
        const x = tensor1d([2, 3]);
        const y = tf.scalar(4);
        const out = tf.einsum('i,->i', x, y);
        expectArraysClose(await out.data(), [8, 12]);
    });
    it('1d reduce sum', async () => {
        const x = tensor1d([2, 4, 6]);
        const out = tf.einsum('i->', x);
        expectArraysClose(await out.data(), 12);
    });
    it('2d matrix reduce sum', async () => {
        const x = tensor2d([[1, 2], [3, 4]]);
        const out = tf.einsum('ij->', x);
        expectArraysClose(await out.data(), 10);
    });
    it('2d matrices multiply and reduce summing', async () => {
        const x = tensor2d([[1, 2], [3, 4]]);
        const y = tensor2d([[4, 3], [2, 1]]);
        const out = tf.einsum('ij,ji->', x, y);
        expectArraysClose(await out.data(), 21);
    });
    it('2d matrix times scalar and reduce summing', async () => {
        const x = tensor2d([[1, 2], [3, 4]]);
        const y = tf.scalar(5);
        const out = tf.einsum('ij,->', x, y);
        expectArraysClose(await out.data(), 50);
    });
    it('two 1d tensors dot', async () => {
        const x = tensor1d([1, 3, 5]);
        const y = tensor1d([2, 4, 6]);
        const out = tf.einsum('i,i->', x, y);
        expectArraysClose(await out.data(), 44);
    });
    it('two 1d tensors outer', async () => {
        const x = tensor1d([1, 3, 5]);
        const y = tensor1d([2, 4, 6]);
        const out = tf.einsum('i,j->ij', x, y);
        expectArraysClose(await out.data(), [[2, 4, 6], [6, 12, 18], [10, 20, 30]]);
    });
    it('2d matrix calculate trace: duplicate axes not implemented yet', () => {
        const x = tensor2d([[1, 2], [3, 4]]);
        expect(() => tf.einsum('ii->', x)).toThrowError(/not implemented yet/);
    });
    it('2d and 1d matrix & vector multiply', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const y = tensor1d([2, 4, 6]);
        const out = tf.einsum('ij,j->i', x, y);
        expectArraysClose(await out.data(), [28, 64]);
    });
    it('2d matrix sum along columns', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const out = tf.einsum('ij->j', x);
        expectArraysClose(await out.data(), [5, 7, 9]);
    });
    it('2d matrix sum along rows', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const out = tf.einsum('ij->i', x);
        expectArraysClose(await out.data(), [6, 15]);
    });
    it('2d matrix transposing', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const out = tf.einsum('ij->ji', x);
        expectArraysClose(await out.data(), [[1, 4], [2, 5], [3, 6]]);
    });
    it('2d matrix multiply', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const y = tensor2d([[0, 1], [2, 3], [4, 5]]);
        const out = tf.einsum('ij,jk->ik', x, y);
        expectArraysClose(await out.data(), [[16, 22], [34, 49]]);
    });
    it('2d matrix multiply and transposing', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const y = tensor2d([[0, 1], [2, 3], [4, 5]]);
        const out = tf.einsum('ij,jk->ki', x, y);
        expectArraysClose(await out.data(), [[16, 34], [22, 49]]);
    });
    it('two 2d matrices batch dot', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const y = tensor2d([[0, 1, 2], [3, 4, 5]]);
        const out = tf.einsum('bi,bi->b', x, y);
        expectArraysClose(await out.data(), [8, 62]);
    });
    it('two 2d matrices batch outer', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const y = tensor2d([[0, 1, 2], [3, 4, 5]]);
        const out = tf.einsum('bi,bj->bij', x, y);
        expectArraysClose(await out.data(), [
            [[0, 1, 2], [0, 2, 4], [0, 3, 6]],
            [[12, 16, 20], [15, 20, 25], [18, 24, 30]]
        ]);
    });
    it('two 3d tensors batch matmul', async () => {
        const x = tf.reshape(tf.range(1, 13), [2, 2, 3]);
        const y = tf.reshape(tf.range(1, 19), [2, 3, 3]);
        const out = tf.einsum('bij,bjk->bik', x, y);
        expectArraysClose(await out.data(), [[[30, 36, 42], [66, 81, 96]], [[318, 342, 366], [435, 468, 501]]]);
    });
    it('two 3d tensors A', async () => {
        const x = tf.reshape(tf.range(1, 9), [2, 2, 2]);
        const y = tf.reshape(tf.range(1, 13), [2, 3, 2]);
        const out = tf.einsum('adc,abc->abd', x, y);
        expectArraysClose(await out.data(), [[[5, 11], [11, 25], [17, 39]], [[83, 113], [105, 143], [127, 173]]]);
    });
    it('two 3d tensors B', async () => {
        const x = tf.reshape(tf.range(1, 9), [2, 2, 2]);
        const y = tf.reshape(tf.range(1, 13), [2, 3, 2]);
        const out = tf.einsum('adc,abc->adb', x, y);
        expectArraysClose(await out.data(), [[[5, 11, 17], [11, 25, 39]], [[83, 105, 127], [113, 143, 173]]]);
    });
    it('one 3d tensor: batch matrix transposing', async () => {
        const x = tensor3d([[[1, 2], [3, 4]], [[-1, -2], [-3, -4]]]);
        const out = tf.einsum('bij->bji', x);
        expectArraysClose(await out.data(), [[[1, 3], [2, 4]], [[-1, -3], [-2, -4]]]);
    });
    it('4d tensor and 3d tensor, contracting two dimensions', async () => {
        const x = tf.reshape(tf.range(1, 33), [2, 4, 2, 2]);
        const y = tf.reshape(tf.range(1, 9), [2, 2, 2]);
        const out = tf.einsum('abcd,cde->abe', x, y);
        expectArraysClose(await out.data(), [
            [[50, 60], [114, 140], [178, 220], [242, 300]],
            [[306, 380], [370, 460], [434, 540], [498, 620]]
        ]);
    });
    it('two 4d tensors, contracting one dimension', async () => {
        const x = tf.reshape(tf.range(1, 33), [2, 4, 2, 2]);
        const y = tf.reshape(tf.range(1, 25), [2, 3, 2, 2]);
        const out = tf.einsum('aecd,abcd->acbe', x, y);
        expectArraysClose(await out.data(), [
            [
                [[5, 17, 29, 41], [17, 61, 105, 149], [29, 105, 181, 257]],
                [[25, 53, 81, 109], [53, 113, 173, 233], [81, 173, 265, 357]]
            ],
            [
                [[473, 581, 689, 797], [613, 753, 893, 1033], [753, 925, 1097, 1269]],
                [[605, 729, 853, 977], [761, 917, 1073, 1229], [917, 1105, 1293, 1481]]
            ]
        ]);
    });
    it('two 4d tensors, contracting two dimensions', async () => {
        const x = tf.reshape(tf.range(1, 33), [2, 4, 2, 2]);
        const y = tf.reshape(tf.range(1, 25), [2, 3, 2, 2]);
        const out = tf.einsum('aecd,abcd->abe', x, y);
        expectArraysClose(await out.data(), [
            [[30, 70, 110, 150], [70, 174, 278, 382], [110, 278, 446, 614]],
            [
                [1078, 1310, 1542, 1774], [1374, 1670, 1966, 2262],
                [1670, 2030, 2390, 2750]
            ]
        ]);
    });
    it('mismatched dimensions throws error', () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const y = tensor2d([[0, 1], [2, 3]]);
        expect(() => tf.einsum('ij,jk->ik', x, y))
            .toThrowError('Expected dimension 3 at axis 0 of input shaped [2,2], ' +
            'but got dimension 2');
    });
    it('incorrect equation throws error', () => {
        const x = tensor2d([[1, 2], [3, 4]]);
        const y = tensor2d([[0, 1], [2, 3]]);
        expect(() => tf.einsum('', x, y))
            .toThrowError('Equations without an arrow are not supported.');
        expect(() => tf.einsum('ij,jk>ik', x, y))
            .toThrowError('Equations without an arrow are not supported.');
    });
    it('incorrect number of tensors throws error', () => {
        const x = tensor2d([[1, 2], [3, 4]]);
        const y = tensor2d([[0, 1], [2, 3]]);
        expect(() => tf.einsum('ij->ji', x, y))
            .toThrowError('Expected 1 input tensors, received 2');
    });
    it('more than two input tensors throws error', async () => {
        const x = tensor2d([[1, 2], [3, 4]]);
        const y = tensor2d([[0, 1], [2, 3]]);
        const z = tensor2d([[-1, 0], [1, 2]]);
        expect(() => tf.einsum('ij,jk,kl->il', x, y, z))
            .toThrowError(/more than 2 input tensors/);
    });
    it('nonexistent dimension throws error', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const y = tensor2d([[0, 1], [2, 3], [4, 5]]);
        expect(() => tf.einsum('ij,jk->in', x, y))
            .toThrowError('Output subscripts contain the label n not present in ' +
            'the input subscripts.');
    });
    it('two arrows in equation throws error', async () => {
        const x = tensor2d([[1, 2, 3], [4, 5, 6]]);
        const y = tensor2d([[0, 1], [2, 3], [4, 5]]);
        expect(() => tf.einsum('ij,jk->ik->i', x, y))
            .toThrowError(/exactly one arrow/);
    });
});
//# sourceMappingURL=data:application/json;base64,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