import { convertToTensor } from '../tensor_util_env';
import * as util from '../util';
import { conv2d } from './conv2d';
import * as conv_util from './conv_util';
import { op } from './operation';
import { reshape } from './reshape';
/**
 * Computes a 1D convolution over the input x.
 *
 * @param x The input tensor, of rank 3 or rank 2, of shape
 *     `[batch, width, inChannels]`. If rank 2, batch of 1 is assumed.
 * @param filter The filter, rank 3, of shape
 *     `[filterWidth, inDepth, outDepth]`.
 * @param stride The number of entries by which the filter is moved right at
 *     each step.
 * @param pad The type of padding algorithm.
 *    - `same` and stride 1: output will be of same size as input,
 *       regardless of filter size.
 *    - `valid`: output will be smaller than input if filter is larger
 *       than 1x1.
 *   - For more info, see this guide:
 *     [https://www.tensorflow.org/api_docs/python/tf/nn/convolution](
 *          https://www.tensorflow.org/api_docs/python/tf/nn/convolution)
 * @param dataFormat An optional string from "NWC", "NCW". Defaults to "NWC",
 *     the data is stored in the order of [batch, in_width, in_channels]. Only
 *     "NWC" is currently supported.
 * @param dilation The dilation rate in which we sample input values in
 *     atrous convolution. Defaults to `1`. If it is greater than 1, then
 *     stride must be `1`.
 * @param dimRoundingMode A string from: 'ceil', 'round', 'floor'. If none is
 *     provided, it will default to truncate.
 *
 * @doc {heading: 'Operations', subheading: 'Convolution'}
 */
function conv1d_(x, filter, stride, pad, dataFormat = 'NWC', dilation = 1, dimRoundingMode) {
    const $x = convertToTensor(x, 'x', 'conv1d');
    const $filter = convertToTensor(filter, 'filter', 'conv1d');
    let x3D = $x;
    let reshapedTo3D = false;
    if ($x.rank === 2) {
        reshapedTo3D = true;
        x3D = reshape($x, [1, $x.shape[0], $x.shape[1]]);
    }
    util.assert(x3D.rank === 3, () => `Error in conv1d: input must be rank 3, but got rank ${x3D.rank}.`);
    util.assert($filter.rank === 3, () => `Error in conv1d: filter must be rank 3, but got rank ` +
        `${$filter.rank}.`);
    conv_util.checkPadOnDimRoundingMode('conv1d', pad, dimRoundingMode);
    util.assert(x3D.shape[2] === $filter.shape[1], () => `Error in conv1d: depth of input (${x3D.shape[2]}) must match ` +
        `input depth for filter ${$filter.shape[1]}.`);
    util.assert(conv_util.eitherStridesOrDilationsAreOne(stride, dilation), () => 'Error in conv1D: Either stride or dilation must be 1. ' +
        `Got stride ${stride} and dilation '${dilation}'`);
    util.assert(dataFormat === 'NWC', () => `Error in conv1d: got dataFormat of ${dataFormat} but only NWC is currently supported.`);
    const filter4D = reshape($filter, [1, $filter.shape[0], $filter.shape[1], $filter.shape[2]]);
    const input4D = reshape(x3D, [x3D.shape[0], 1, x3D.shape[1], x3D.shape[2]]);
    const strides = [1, stride];
    const dilations = [1, dilation];
    const conv2dDataFormat = 'NHWC';
    const res = conv2d(input4D, filter4D, strides, pad, conv2dDataFormat, dilations, dimRoundingMode);
    if (reshapedTo3D) {
        return reshape(res, [res.shape[2], res.shape[3]]);
    }
    return reshape(res, [res.shape[0], res.shape[2], res.shape[3]]);
}
export const conv1d = op({ conv1d_ });
//# sourceMappingURL=data:application/json;base64,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