/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from './index';
import { ALL_ENVS, describeWithFlags } from './jasmine_util';
import { expectArraysClose } from './test_util';
describeWithFlags('kernel_registry', ALL_ENVS, () => {
    it('register a kernel and call it', () => {
        // Make sure the backend is loaded. Perhaps tf.getBackend
        // should call tf.backend to make sure the backend is loaded?
        expect(tf.backend()).toBeDefined();
        let called = false;
        tf.registerKernel({
            kernelName: 'MyKernel',
            backendName: tf.getBackend(),
            kernelFunc: ({ inputs, attrs }) => {
                expect(attrs.a).toBe(5);
                expect(inputs.x.shape).toEqual([2, 2]);
                expect(inputs.x.dtype).toBe('float32');
                called = true;
                return { dtype: 'float32', shape: [3, 3], dataId: {} };
            }
        });
        const inputs = { x: tf.zeros([2, 2]) };
        const attrs = { a: 5 };
        const res = tf.engine().runKernel('MyKernel', inputs, attrs);
        expect(called).toBe(true);
        expect(res.dtype).toBe('float32');
        expect(res.shape).toEqual([3, 3]);
        tf.unregisterKernel('MyKernel', tf.getBackend());
    });
    it('errors when running non-existent kernel', () => {
        const inputs = {};
        const attrs = {};
        expect(() => tf.engine().runKernel('DoesNotExist', inputs, attrs))
            .toThrowError();
    });
    // TODO (yassogba) double registration happens now because a backend might be
    // imported more than once (e.g. by a top level package and a dependent
    // package). We may want to remove this test long-term but skip it for
    // now.
    // tslint:disable-next-line: ban
    xit('errors when registering the same kernel twice', () => {
        tf.registerBackend('backend1', () => {
            return {
                id: 1,
                dispose: () => null,
                disposeData: (dataId) => null,
                numDataIds: () => 0
            };
        });
        tf.registerKernel({
            kernelName: 'MyKernel',
            backendName: 'backend1',
            kernelFunc: () => {
                return null;
            }
        });
        expect(() => tf.registerKernel({
            kernelName: 'MyKernel',
            backendName: 'backend1',
            kernelFunc: () => {
                return null;
            }
        })).toThrowError();
        tf.unregisterKernel('MyKernel', 'backend1');
        tf.removeBackend('backend1');
    });
    it('register same kernel on two different backends', () => {
        tf.registerBackend('backend1', () => {
            return {
                id: 1,
                dispose: () => null,
                disposeData: (dataId) => true,
                numDataIds: () => 0
            };
        });
        tf.registerBackend('backend2', () => {
            return {
                id: 2,
                dispose: () => null,
                disposeData: (dataId) => null,
                numDataIds: () => 0
            };
        });
        let lastStorageId = -1;
        const kernelFunc = ({ backend }) => {
            lastStorageId = backend.id;
            return { dataId: {}, shape: [], dtype: 'float32' };
        };
        tf.registerKernel({ kernelName: 'MyKernel', backendName: 'backend1', kernelFunc });
        tf.registerKernel({ kernelName: 'MyKernel', backendName: 'backend2', kernelFunc });
        // No kernel has been executed yet.
        expect(lastStorageId).toBe(-1);
        // Kernel was executed on the first backend.
        tf.setBackend('backend1');
        tf.engine().runKernel('MyKernel', {}, {});
        expect(lastStorageId).toBe(1);
        // Kernel was executed on the second backend.
        tf.setBackend('backend2');
        tf.engine().runKernel('MyKernel', {}, {});
        expect(lastStorageId).toBe(2);
        tf.removeBackend('backend1');
        tf.removeBackend('backend2');
        tf.unregisterKernel('MyKernel', 'backend1');
        tf.unregisterKernel('MyKernel', 'backend2');
    });
    it('register kernel with setup and dispose functions', () => {
        const backendName = 'custom-backend';
        const kernelName = 'MyKernel';
        const customBackend = {
            dispose: () => null,
            disposeData: (dataId) => true,
            numDataIds: () => 0
        };
        tf.registerBackend(backendName, () => customBackend);
        const kernelFunc = () => {
            return { dataId: {}, shape: [], dtype: 'float32' };
        };
        let setupCalled = false;
        const setupFunc = (backend) => {
            expect(backend).toBe(customBackend);
            setupCalled = true;
        };
        let disposeCalled = false;
        const disposeFunc = (backend) => {
            expect(backend).toBe(customBackend);
            disposeCalled = true;
        };
        tf.registerKernel({ kernelName, backendName, kernelFunc, setupFunc, disposeFunc });
        expect(setupCalled).toBe(false);
        expect(disposeCalled).toBe(false);
        tf.setBackend(backendName);
        expect(setupCalled).toBe(true);
        expect(disposeCalled).toBe(false);
        // Kernel was executed on the first backend.
        tf.engine().runKernel(kernelName, {}, {});
        tf.removeBackend(backendName);
        expect(setupCalled).toBe(true);
        expect(disposeCalled).toBe(true);
        tf.unregisterKernel(kernelName, backendName);
    });
});
describeWithFlags('gradient registry', ALL_ENVS, () => {
    it('register a kernel with gradient and call it', async () => {
        let kernelWasCalled = false;
        let gradientWasCalled = false;
        const kernelName = 'MyKernel';
        const x = tf.zeros([2, 2]);
        tf.registerKernel({
            kernelName,
            backendName: tf.getBackend(),
            kernelFunc: () => {
                kernelWasCalled = true;
                return { dtype: 'float32', shape: [3, 3], dataId: {} };
            }
        });
        tf.registerGradient({
            kernelName,
            inputsToSave: ['x'],
            gradFunc: (dy, saved) => {
                // Make sure saved input (x) was passed to the gradient function.
                expect(saved[0].dataId).toEqual(x.dataId);
                // Make sure dy matches the shape of the output.
                expect(dy.shape).toEqual([3, 3]);
                gradientWasCalled = true;
                return { x: () => tf.fill([2, 2], 3) };
            },
        });
        const gradFunc = tf.grad(x => tf.engine().runKernel(kernelName, { x }, {} /* attrs */));
        const dx = gradFunc(x);
        expect(kernelWasCalled).toBe(true);
        expect(gradientWasCalled).toBe(true);
        expect(dx.dtype).toBe('float32');
        expect(dx.shape).toEqual([2, 2]);
        expectArraysClose(await dx.data(), [3, 3, 3, 3]);
        tf.unregisterKernel(kernelName, tf.getBackend());
        tf.unregisterGradient(kernelName);
    });
    it('register a kernel with gradient that specifies outputsToSave and call it', async () => {
        let kernelWasCalled = false;
        let gradientWasCalled = false;
        const kernelName = 'MyKernel';
        const tensor = tf.zeros([3, 3], 'float32');
        const forwardReturnDataId = tensor.dataId;
        tf.registerKernel({
            kernelName,
            backendName: tf.getBackend(),
            kernelFunc: () => {
                kernelWasCalled = true;
                return {
                    dtype: tensor.dtype,
                    shape: tensor.shape,
                    dataId: forwardReturnDataId
                };
            }
        });
        tf.registerGradient({
            kernelName,
            outputsToSave: [true],
            gradFunc: (dy, saved) => {
                // Make sure saved output was passed to the gradient function.
                expect(saved[0].dataId).toEqual(forwardReturnDataId);
                // Make sure dy matches the shape of the output.
                expect(dy.shape).toEqual([3, 3]);
                gradientWasCalled = true;
                return { x: () => tf.fill([2, 2], 3) };
            },
        });
        const gradFunc = tf.grad(x => tf.engine().runKernel(kernelName, { x }, {} /* attrs */));
        const x = tf.zeros([2, 2]);
        const dx = gradFunc(x);
        expect(kernelWasCalled).toBe(true);
        expect(gradientWasCalled).toBe(true);
        expect(dx.dtype).toBe('float32');
        expect(dx.shape).toEqual([2, 2]);
        tf.unregisterKernel(kernelName, tf.getBackend());
        tf.unregisterGradient(kernelName);
    });
    it('register a kernel with array inputs and saveAllInputs true', async () => {
        let kernelWasCalled = false;
        let gradientWasCalled = false;
        const kernelName = 'MyKernel';
        const x = [tf.zeros([2, 2]), tf.zeros([2, 2])];
        const forwardReturnDataId = {};
        tf.registerKernel({
            kernelName,
            backendName: tf.getBackend(),
            kernelFunc: () => {
                kernelWasCalled = true;
                return { dtype: 'float32', shape: [3, 3], dataId: forwardReturnDataId };
            }
        });
        tf.registerGradient({
            kernelName,
            saveAllInputs: true,
            gradFunc: (dy, saved) => {
                // Make sure saved input (x) was passed to the gradient function.
                const [$x0, $x1] = x;
                expect(saved.length).toEqual(x.length);
                expect($x0.dataId).toEqual(x[0].dataId);
                expect($x1.dataId).toEqual(x[1].dataId);
                gradientWasCalled = true;
                return { 0: () => tf.fill([2, 2], 3), 1: () => tf.fill([2, 2], 3) };
            }
        });
        // Inputs as array.
        const z = (...x) => 
        // tslint:disable-next-line: no-unnecessary-type-assertion
        tf.engine().runKernel(kernelName, x, {} /* attrs */);
        const gradFunc = tf.grads(z);
        const dx = gradFunc(x);
        expect(kernelWasCalled).toBe(true);
        expect(gradientWasCalled).toBe(true);
        expect(dx.length).toEqual(2);
        expect(dx[0].dtype).toBe('float32');
        expect(dx[0].shape).toEqual([2, 2]);
        expect(dx[1].dtype).toBe('float32');
        expect(dx[1].shape).toEqual([2, 2]);
        expectArraysClose(await dx[0].data(), [3, 3, 3, 3]);
        expectArraysClose(await dx[1].data(), [3, 3, 3, 3]);
        tf.unregisterKernel(kernelName, tf.getBackend());
        tf.unregisterGradient(kernelName);
    });
    it('register a kernel with map inputs and saveAllInputs true should throw ' +
        'error', async () => {
        const kernelName = 'MyKernel';
        const x0 = tf.zeros([2, 2]);
        const x1 = tf.zeros([2, 2]);
        const forwardReturnDataId = {};
        tf.registerKernel({
            kernelName,
            backendName: tf.getBackend(),
            kernelFunc: () => {
                return {
                    dtype: 'float32',
                    shape: [3, 3],
                    dataId: forwardReturnDataId
                };
            }
        });
        tf.registerGradient({
            kernelName,
            saveAllInputs: true,
            gradFunc: (dy, saved) => {
                // Make sure saved input (x) was passed to the gradient function.
                const [$x0, $x1] = saved;
                expect($x0.dataId).toEqual(x0.dataId);
                expect($x1.dataId).toEqual(x1.dataId);
                return { x0: () => tf.fill([2, 2], 3), x1: () => tf.fill([2, 2], 3) };
            }
        });
        // Inputs as map.
        const z = (x0, x1) => 
        // tslint:disable-next-line: no-unnecessary-type-assertion
        tf.engine().runKernel(kernelName, { x0, x1 }, {} /* attrs */);
        const gradFunc = tf.grads(z);
        expect(() => gradFunc([x0, x1]))
            .toThrowError(/saveAllInputs is true, expected inputs to be an array/);
        tf.unregisterKernel(kernelName, tf.getBackend());
        tf.unregisterGradient(kernelName);
    });
    it('errors when running non-existent gradient', () => {
        const kernelName = 'MyKernel';
        const x = tf.zeros([2, 2]);
        tf.registerKernel({
            kernelName,
            backendName: tf.getBackend(),
            kernelFunc: () => ({ dtype: 'float32', shape: [3, 3], dataId: {} })
        });
        const gradFunc = tf.grad(x => tf.engine().runKernel(kernelName, { x }, {} /* attrs */));
        expect(() => gradFunc(x))
            .toThrowError(/gradient function not found for MyKernel/);
        tf.unregisterKernel(kernelName, tf.getBackend());
    });
    // tslint:disable-next-line: ban
    xit('warning when registering the same gradient twice', () => {
        const kernelName = 'MyKernel';
        tf.registerGradient({ kernelName, gradFunc: () => null });
        spyOn(console, 'warn').and.callFake((msg) => {
            expect(msg).toBe('Overriding the gradient for \'MyKernel\'');
        });
        tf.registerGradient({ kernelName, gradFunc: () => null });
        tf.unregisterGradient(kernelName);
    });
});
//# sourceMappingURL=data:application/json;base64,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