/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { isPromise } from './util_base';
// Expects flags from URL in the format ?tfjsflags=FLAG1:1,FLAG2:true.
const TENSORFLOWJS_FLAGS_PREFIX = 'tfjsflags';
/**
 * The environment contains evaluated flags as well as the registered platform.
 * This is always used as a global singleton and can be retrieved with
 * `tf.env()`.
 *
 * @doc {heading: 'Environment'}
 */
export class Environment {
    // tslint:disable-next-line: no-any
    constructor(global) {
        this.global = global;
        this.flags = {};
        this.flagRegistry = {};
        this.urlFlags = {};
        // Jasmine spies on this in 'environment_test.ts'
        this.getQueryParams = getQueryParams;
        this.populateURLFlags();
    }
    setPlatform(platformName, platform) {
        if (this.platform != null) {
            if (!(env().getBool('IS_TEST') || env().getBool('PROD'))) {
                console.warn(`Platform ${this.platformName} has already been set. ` +
                    `Overwriting the platform with ${platform}.`);
            }
        }
        this.platformName = platformName;
        this.platform = platform;
    }
    registerFlag(flagName, evaluationFn, setHook) {
        this.flagRegistry[flagName] = { evaluationFn, setHook };
        // Override the flag value from the URL. This has to happen here because
        // the environment is initialized before flags get registered.
        if (this.urlFlags[flagName] != null) {
            const flagValue = this.urlFlags[flagName];
            if (!(env().getBool('IS_TEST') || env().getBool('PROD'))) {
                console.warn(`Setting feature override from URL ${flagName}: ${flagValue}.`);
            }
            this.set(flagName, flagValue);
        }
    }
    async getAsync(flagName) {
        if (flagName in this.flags) {
            return this.flags[flagName];
        }
        this.flags[flagName] = await this.evaluateFlag(flagName);
        return this.flags[flagName];
    }
    get(flagName) {
        if (flagName in this.flags) {
            return this.flags[flagName];
        }
        const flagValue = this.evaluateFlag(flagName);
        if (isPromise(flagValue)) {
            throw new Error(`Flag ${flagName} cannot be synchronously evaluated. ` +
                `Please use getAsync() instead.`);
        }
        this.flags[flagName] = flagValue;
        return this.flags[flagName];
    }
    getNumber(flagName) {
        return this.get(flagName);
    }
    getBool(flagName) {
        return this.get(flagName);
    }
    getFlags() {
        return this.flags;
    }
    // For backwards compatibility.
    get features() {
        return this.flags;
    }
    set(flagName, value) {
        if (this.flagRegistry[flagName] == null) {
            throw new Error(`Cannot set flag ${flagName} as it has not been registered.`);
        }
        this.flags[flagName] = value;
        if (this.flagRegistry[flagName].setHook != null) {
            this.flagRegistry[flagName].setHook(value);
        }
    }
    evaluateFlag(flagName) {
        if (this.flagRegistry[flagName] == null) {
            throw new Error(`Cannot evaluate flag '${flagName}': no evaluation function found.`);
        }
        return this.flagRegistry[flagName].evaluationFn();
    }
    setFlags(flags) {
        this.flags = Object.assign({}, flags);
    }
    reset() {
        this.flags = {};
        this.urlFlags = {};
        this.populateURLFlags();
    }
    populateURLFlags() {
        if (typeof this.global === 'undefined' ||
            typeof this.global.location === 'undefined' ||
            typeof this.global.location.search === 'undefined') {
            return;
        }
        const urlParams = this.getQueryParams(this.global.location.search);
        if (TENSORFLOWJS_FLAGS_PREFIX in urlParams) {
            const keyValues = urlParams[TENSORFLOWJS_FLAGS_PREFIX].split(',');
            keyValues.forEach(keyValue => {
                const [key, value] = keyValue.split(':');
                this.urlFlags[key] = parseValue(key, value);
            });
        }
    }
}
export function getQueryParams(queryString) {
    const params = {};
    queryString.replace(/[?&]([^=?&]+)(?:=([^&]*))?/g, (s, ...t) => {
        decodeParam(params, t[0], t[1]);
        return t.join('=');
    });
    return params;
}
function decodeParam(params, name, value) {
    params[decodeURIComponent(name)] = decodeURIComponent(value || '');
}
function parseValue(flagName, value) {
    value = value.toLowerCase();
    if (value === 'true' || value === 'false') {
        return value === 'true';
    }
    else if (`${+value}` === value) {
        return +value;
    }
    throw new Error(`Could not parse value flag value ${value} for flag ${flagName}.`);
}
/**
 * Returns the current environment (a global singleton).
 *
 * The environment object contains the evaluated feature values as well as the
 * active platform.
 *
 * @doc {heading: 'Environment'}
 */
export function env() {
    return ENV;
}
export let ENV = null;
export function setEnvironmentGlobal(environment) {
    ENV = environment;
}
//# sourceMappingURL=data:application/json;base64,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