"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.trackNetworkEvents = exports.shouldTrackNetworkEvent = void 0;
var tslib_1 = require("tslib");
var analytics_core_1 = require("@amplitude/analytics-core");
var rxjs_1 = require("rxjs");
var constants_1 = require("./constants");
var DEFAULT_STATUS_CODE_RANGE = '500-599';
function wildcardMatch(str, pattern) {
    // Escape all regex special characters except for *
    var escapedPattern = pattern.replace(/[-[\]{}()+?.,\\^$|#\s]/g, '\\$&');
    // Replace * with .*
    var regexPattern = '^' + escapedPattern.replace(/\*/g, '.*') + '$';
    var regex = new RegExp(regexPattern);
    return regex.test(str);
}
function isStatusCodeInRange(statusCode, range) {
    var e_1, _a;
    var ranges = range.split(',');
    try {
        for (var ranges_1 = tslib_1.__values(ranges), ranges_1_1 = ranges_1.next(); !ranges_1_1.done; ranges_1_1 = ranges_1.next()) {
            var r = ranges_1_1.value;
            var _b = tslib_1.__read(r.split('-').map(Number), 2), start = _b[0], end = _b[1];
            if (statusCode === start && end === undefined) {
                return true;
            }
            if (statusCode >= start && statusCode <= end) {
                return true;
            }
        }
    }
    catch (e_1_1) { e_1 = { error: e_1_1 }; }
    finally {
        try {
            if (ranges_1_1 && !ranges_1_1.done && (_a = ranges_1.return)) _a.call(ranges_1);
        }
        finally { if (e_1) throw e_1.error; }
    }
    return false;
}
function isCaptureRuleMatch(rule, hostname, status) {
    // check if the host is in the allowed hosts
    if (rule.hosts && !rule.hosts.find(function (host) { return wildcardMatch(hostname, host); })) {
        return;
    }
    // check if the status code is in the allowed range
    if (status || status === 0) {
        var statusCodeRange = rule.statusCodeRange || DEFAULT_STATUS_CODE_RANGE;
        if (!isStatusCodeInRange(status, statusCodeRange)) {
            return false;
        }
    }
    return true;
}
function parseUrl(url) {
    var _a;
    if (!url) {
        return;
    }
    try {
        /* istanbul ignore next */
        var currentHref = (_a = (0, analytics_core_1.getGlobalScope)()) === null || _a === void 0 ? void 0 : _a.location.href;
        var urlObj = new URL(url, currentHref);
        var query = urlObj.searchParams.toString();
        var fragment = urlObj.hash.replace('#', '');
        var href = urlObj.href;
        var host = urlObj.host;
        urlObj.hash = '';
        urlObj.search = '';
        var hrefWithoutQueryOrHash = urlObj.href;
        return { query: query, fragment: fragment, href: href, hrefWithoutQueryOrHash: hrefWithoutQueryOrHash, host: host };
    }
    catch (e) {
        /* istanbul ignore next */
        return;
    }
}
function shouldTrackNetworkEvent(networkEvent, options) {
    var _a;
    if (options === void 0) { options = {}; }
    var urlObj = parseUrl(networkEvent.url);
    /* istanbul ignore if */
    if (!urlObj) {
        // if the URL failed to parse, do not track the event
        // this is a probably impossible case that would only happen if the URL is malformed
        /* istanbul ignore next */
        return false;
    }
    var host = urlObj.host;
    // false if is amplitude request and not configured to track amplitude requests
    if (options.ignoreAmplitudeRequests !== false &&
        (wildcardMatch(host, '*.amplitude.com') || wildcardMatch(host, 'amplitude.com'))) {
        return false;
    }
    // false if the host is in the ignore list
    if ((_a = options.ignoreHosts) === null || _a === void 0 ? void 0 : _a.find(function (ignoreHost) { return wildcardMatch(host, ignoreHost); })) {
        return false;
    }
    // false if the status code is not 500-599 and there are no captureRules
    if (!options.captureRules &&
        networkEvent.status !== undefined &&
        !isStatusCodeInRange(networkEvent.status, DEFAULT_STATUS_CODE_RANGE)) {
        return false;
    }
    if (options.captureRules) {
        // find the first capture rule, in reverse-order,
        // that is a match (true) or a miss (false)
        var isMatch_1;
        tslib_1.__spreadArray([], tslib_1.__read(options.captureRules), false).reverse().find(function (rule) {
            isMatch_1 = isCaptureRuleMatch(rule, host, networkEvent.status);
            return isMatch_1 !== undefined;
        });
        // if we found a miss (false) or no match (undefined),
        // then do not track the event
        if (!isMatch_1) {
            return false;
        }
    }
    return true;
}
exports.shouldTrackNetworkEvent = shouldTrackNetworkEvent;
function trackNetworkEvents(_a) {
    var allObservables = _a.allObservables, networkTrackingOptions = _a.networkTrackingOptions, amplitude = _a.amplitude;
    var networkObservable = allObservables.networkObservable;
    var filteredNetworkObservable = networkObservable.pipe((0, rxjs_1.filter)(function (event) {
        // Only track network events that should be tracked,
        return shouldTrackNetworkEvent(event.event, networkTrackingOptions);
    }));
    return filteredNetworkObservable.subscribe(function (networkEvent) {
        var _a;
        var _b, _c;
        var request = networkEvent.event;
        // convert to NetworkAnalyticsEvent
        var urlObj = parseUrl(request.url);
        /* istanbul ignore if */
        if (!urlObj) {
            // if the URL failed to parse, do not track the event
            // this is a very unlikely case, because URL() shouldn't throw an exception
            // when the URL is a valid URL
            /* istanbul ignore next */
            return;
        }
        var responseBodySize = (_b = request.responseWrapper) === null || _b === void 0 ? void 0 : _b.bodySize;
        /* istanbul ignore next */
        var requestBodySize = (_c = request.requestWrapper) === null || _c === void 0 ? void 0 : _c.bodySize;
        var networkAnalyticsEvent = (_a = {},
            _a['[Amplitude] URL'] = urlObj.hrefWithoutQueryOrHash,
            _a['[Amplitude] URL Query'] = urlObj.query,
            _a['[Amplitude] URL Fragment'] = urlObj.fragment,
            _a['[Amplitude] Request Method'] = request.method,
            _a['[Amplitude] Status Code'] = request.status,
            _a['[Amplitude] Start Time'] = request.startTime,
            _a['[Amplitude] Completion Time'] = request.endTime,
            _a['[Amplitude] Duration'] = request.duration,
            _a['[Amplitude] Request Body Size'] = requestBodySize,
            _a['[Amplitude] Response Body Size'] = responseBodySize,
            _a['[Amplitude] Request Type'] = request.type,
            _a);
        /* istanbul ignore next */
        amplitude === null || amplitude === void 0 ? void 0 : amplitude.track(constants_1.AMPLITUDE_NETWORK_REQUEST_EVENT, networkAnalyticsEvent);
    });
}
exports.trackNetworkEvents = trackNetworkEvents;
//# sourceMappingURL=track-network-event.js.map