import { AllureCommandStepExecutable } from "./AllureCommandStep.js";
import { ExecutableItemWrapper } from "./ExecutableItemWrapper.js";
import { testResult } from "./constructors.js";
import { LinkType } from "./model.js";
import { getLabelsFromEnv, md5 } from "./utils.js";
export class AllureTest extends ExecutableItemWrapper {
    runtime;
    testResult;
    historyIdSetManually = false;
    constructor(runtime, start = Date.now()) {
        super(testResult());
        this.runtime = runtime;
        const globalLabels = getLabelsFromEnv();
        this.testResult = this.wrappedItem;
        this.testResult.start = start;
        globalLabels.forEach((label) => this.addLabel(label.name, label.value));
    }
    endTest(stop = Date.now()) {
        this.testResult.stop = stop;
        this.runtime.writeResult(this.testResult);
    }
    get uuid() {
        return this.testResult.uuid;
    }
    set historyId(id) {
        this.historyIdSetManually = true;
        this.testResult.historyId = id;
    }
    set fullName(fullName) {
        this.testResult.fullName = fullName;
    }
    set testCaseId(testCaseId) {
        this.testResult.testCaseId = testCaseId;
    }
    addLabel(name, value) {
        this.testResult.labels.push({ name, value });
    }
    addLink(url, name, type) {
        this.testResult.links.push({ name, url, type });
    }
    addIssueLink(url, name) {
        this.addLink(url, name, LinkType.ISSUE);
    }
    addTmsLink(url, name) {
        this.addLink(url, name, LinkType.TMS);
    }
    calculateHistoryId() {
        if (this.historyIdSetManually) {
            return;
        }
        const tcId = this.testResult.testCaseId
            ? this.testResult.testCaseId
            : this.testResult.fullName
                ? md5(this.testResult.fullName)
                : null;
        if (!tcId) {
            return;
        }
        const paramsString = this.testResult.parameters
            .filter((p) => !p?.excluded)
            .sort((a, b) => a.name?.localeCompare(b?.name) || a.value?.localeCompare(b?.value))
            .map((p) => `${p.name ?? "null"}:${p.value ?? "null"}`)
            .join(",");
        const paramsHash = md5(paramsString);
        this.historyId = `${tcId}:${paramsHash}`;
    }
    applyMetadata(metadata, stepApplyFn) {
        const { attachments = [], labels = [], links = [], parameter = [], steps = [], description, descriptionHtml, displayName, historyId, testCaseId, } = metadata;
        labels.forEach((label) => {
            this.addLabel(label.name, label.value);
        });
        links.forEach((link) => {
            this.addLink(link.url, link.name, link.type);
        });
        parameter.forEach((param) => {
            this.parameter(param.name, param.value, {
                excluded: param.excluded,
                mode: param.mode,
            });
        });
        attachments.forEach((attachment) => {
            const attachmentFilename = this.runtime.writeAttachment(attachment.content, attachment.type, attachment.encoding);
            this.addAttachment(attachment.name, {
                contentType: attachment.type,
            }, attachmentFilename);
        });
        if (description) {
            this.description = description;
        }
        if (descriptionHtml) {
            this.descriptionHtml = descriptionHtml;
        }
        if (displayName) {
            this.name = displayName;
        }
        if (testCaseId) {
            this.testCaseId = testCaseId;
        }
        if (historyId) {
            this.historyId = historyId;
        }
        steps.forEach((stepMetadata) => {
            const step = AllureCommandStepExecutable.toExecutableItem(this.runtime, stepMetadata);
            if (stepApplyFn) {
                stepApplyFn(step);
                return;
            }
            this.addStep(step);
        });
    }
}
//# sourceMappingURL=AllureTest.js.map